/* sc.h - Smartcard related functions
 *
 * Copyright 1993-1997, Tim Hudson. All rights reserved.
 *
 * You can pretty much do what you like with this code except pretend that 
 * you wrote it provided that any derivative of this code includes the
 * above comments unchanged. If you put this in a product then attribution
 * is mandatory. See the details in the COPYING file.
 *
 * Tim Hudson
 * tjh@cryptsoft.com
 *
 */

#ifndef HEADER_SC_H
#define HEADER_SC_H

#include "platform.h"
#include "sio.h"
#include "slog.h"

/* current version - 1.4 */
#define SC_VERSION_MAJOR        1
#define SC_VERSION_MINOR        4

int SC_GetVersion(int *vmajor, int *vminor);

/* macros modelled off the select FD_SET stuff for testing large
 * bitsets "efficiently" - well at least you don't have to pass
 * around 64k arrays
 *
 * This is useful in the file system scanning logic in particular
 * and is something that is pending implementation - i.e. the existing
 * logic doesn't use any of this stuff just yet
 */
#define SC_SETSIZE      (64*1024L)
#define SC_NBBY         8
typedef long sc_mask;
#define SC_NFDBITS (sizeof(sc_mask)*SC_NBBY)
#define sc_howmany(x, y)   (((x)+((y)-1))/(y))
typedef struct sc_set {
  sc_mask scs_bits[sc_howmany(SC_SETSIZE, SC_NFDBITS)];
} sc_set;
#define SC_SET(n, p)    ((p)->scs_bits[(n)/SC_NFDBITS] |= \
                            ((unsigned)1 << ((n) % SC_NFDBITS)))
#define SC_CLR(n, p)    ((p)->scs_bits[(n)/NFDBITS] &= \
                            ~((unsigned)1 << ((n) % SC_NFDBITS)))
 
#define SC_ISSET(n, p)  ((p)->scs_bits[(n)/SC_NFDBITS] & \
                            ((unsigned)1 << ((n) % SC_NFDBITS)))
#define SC_ZERO(p)      memset((char *)(p), 0, sizeof (*(p)))


/* the ATR can be at most 1+32 characters ... we store the raw data 
 * in "data" and the decoded data in the rest of the fields pulled
 * appart so that we don't have to mess with ATR logic to get at
 * the fields. There is redundant information naturally.
 */
typedef struct {
  unsigned char data[32+1];
  int data_count;

  unsigned char TS;
  unsigned char T0;

  int TCK;
  unsigned char TCK_sent;  /* transmitted TCK */
  unsigned char TCK_calc;  /* calculated TCK */

  /* there are limits to how many of TA[i] you can have but
   * is this documented somewhere - I'm assuming that 3 is 
   * the max for any of the chars
   *
   * note: entry[0] in each of these is ignored so that
   *       TA1 is stored in TA[1] (not TA[0]) so that 
   *       the code matches the documentation
   */
  unsigned char TA[1+3];
  int TA_count;
  unsigned char TB[1+3];
  int TB_count;
  unsigned char TC[1+3];
  int TC_count;

  /* we don't store TD bytes as there isn't anything in them
   * that is of interest except the transmission protocol currently
   * in use which we extract separately - however I might just be
   * wrong as there are cards that can support lots of different 
   * protocols which are probably indicated via multiple TD bytes
   */

  /* historical bytes are implicitly limited by the bounds of
   * the total message size
   */
  char historical[32];
  int historical_count;

  /* now for the redundant interpretation of the above data */
  int direct_convention;
  int inverse_convention;

  /* transmission protocol extracted from TD bytes */
  int transmission_protocol;

} SC_ATR;

/* handle ATR format data */
SC_ATR *SC_DecodeATR(char *buf,int len);
char *SC_ATR2Text(SC_ATR *atr,int flag);

#ifdef USE_STDIO
/* standard ATR diagnostic output */
int SC_DumpATR(FILE *outf, char *buf,int len,int flag);
#endif /* USE_STDIO */

int SC_LogATR(SLOG *outf, char *buf,int len, int flag);

#define SC_READER_UNKNOWN      -1
#define SC_READER_DUMBMOUSE    0        /* dumbmouse */
#define SC_READER_SCT          1        /* Schlumberger SCT series etc */
#define SC_READER_REFLEX       2        /* Schlumberger Reflex series etc */
#define SC_READER_GCR          3        /* Gemplus GCR series */
#define SC_READER_PE	       4	/* Philips PE series */
#define SC_READER_COUNT        5

#define SC_CARD_UNKNOWN        -1
#define SC_CARD_CRYPTOFLEX     0        /* Schlumberger Cryptoflex */
#define SC_CARD_DX             1        /* Philips DX */
#define SC_CARD_CHIPKNIP       2        /* Chipknip */
#define SC_CARD_GSM	       3	/* GSM SIM */
#define SC_CARD_COUNT          4

/* we need a standard mapping for file types */
#define SC_FILETYPE_FILE      0
#define SC_FILETYPE_EF	      0
#define SC_FILETYPE_DIR	      1

/* different PE reader series have different "device" ids 
 * and we are hardwiring to the PE122 series for the moment
 */
#define SC_READER_PE_DEVICE_ID	0x15

/* conversion to and from code/string */
char *SC_Reader2String(int code);
int SC_String2Reader(char *desc);
char *SC_Card2String(int code);
int SC_String2Card(char *desc);

#define SC_MAX_DATA_LEN         256
#define SC_MAX_HEX_DATA_LEN     ((256*3)+1)
#define SC_VALUE_BUF_LEN        (256*2)
#define SC_VALUE_MAX_VARS       32

/* SC_MAX_TRANSFER_LEN -> max bytes read/written in higher-level
 * 			  read/write commands
 */
/*
#define SC_MAX_TRANSFER_LEN	128
*/
#define SC_MAX_TRANSFER_LEN	 64

/* now for the higher-level ISO 7816 handling commands */
typedef struct {
  unsigned char CLA;    /* class */
  unsigned char INS;    /* instruction */
  unsigned char P1;     /* parameter 1 */
  unsigned char P2;     /* parameter 2 */
  int Lc;               /* Length command - length of data */
  int Le;               /* Length expected - expected result length */
  char data[SC_MAX_DATA_LEN];       /* command data */
} SC_CMD;

typedef struct {
  unsigned char SW1;
  unsigned char SW2;
  int len;
  char data[SC_MAX_DATA_LEN];
  int rsp_code;      /* interpreted code */
} SC_RSP;

#define SC_LAST_RSP_CODE(sc) (sc)->last_rsp.rsp_code
#define SC_LAST_RSP_STRING(sc) SC_RSPCode2String((sc)->last_rsp.rsp_code)

typedef struct {
  SIO_INFO *sio;      /* serial device interface */
  SC_ATR *atr;        /* card ATR - optional */
  SLOG *slog;         /* debug/trace log */
  SLOG *sout;         /* output */
  int reader_type;    /* reader protocol type (one of SC_READER_*) */
  int card_type;      /* card type (one of SC_CARD_*) if known */

  int exe_depth;      /* execution depth (0=base,1=first,2=second) */

  int error_code;     /* error code from last "command" */

  /* the results of the last command are kept here
   * for chaining purposes so we can get into the data
   * no matter how we got it (ascii or raw or whatever)
   */
  SC_RSP last_rsp;

  /* parsing interface - data and names are stored here and pointers
   *                     into each for the name and the value are
   *                     how things are accessed. This isn't the nicest
   *                     interface but heaps of mallocs are not on at
   *                     this level of the code!
   */
  char vardata[SC_VALUE_BUF_LEN];
  char valdata[SC_VALUE_BUF_LEN];
  char *vars[SC_VALUE_MAX_VARS];
  char *vals[SC_VALUE_MAX_VARS];
  int N_vars;
  int vardata_used;
  int valdata_used;

  int noexec;            /* disable actual execution of commands */
  int noparse;           /* disable autoparse of command output */

} SC_STATE;

SC_STATE *SC_Init(SIO_INFO *s,int reader_type,int card_type, int probe,
                                                        SLOG *log);
int SC_Final(SC_STATE *sc);
int SC_SetLog(SC_STATE *sc,SLOG *slog);
int SC_SetOutput(SC_STATE *sc,SLOG *sout);
char *SC_ErrorCode2String(int err);
int SC_SetTestMode(SC_STATE *sc,int testval);
int SC_SetParseMode(SC_STATE *sc,int parseval);

/* this is really for library internal use only ... but you might want
 * to call it yourself to check that the card is still in place
 */
int SC_GetATR(SC_STATE *sc);

/* handy routine for when using SC_ExecuteCommand when you want to
 * pull appart the ASCII output in a standard manner 
 */
int SC_HexToRSP(char *inbuf, SC_RSP *uout);

/* command direction -> basically OUT means that Lc is really Le and
 * there is no data sent with the command ... try and find that 
 * documented clearly anywhere :-)
 */
#define SC_DIR_IN  0
#define SC_DIR_OUT 1

int SC_DoAsciiCommand(SC_STATE *sc,int direction,char *uin,char *uout);
int SC_DoCommand(SC_STATE *sc,int direction, SC_CMD *uin, SC_RSP *uout);

/* conversion routines in scasc.c */
unsigned char SC_ByteToHex(unsigned char c);
unsigned char SC_HexToByte(unsigned char c);
int SC_HexToBuffer(char *inbuf,char *outbuf,int maxsize);
int SC_BufferToHex(char *inbuf,int inlen,char *outbuf,int maxsize,int flags);

/* card specific command handling in sccards.c */
int SC_ExecuteCommand(SC_STATE *sc,char *cmd,char *uout,int chain);

/* execute a stream of commands (stops if any of the commands fail) */
int SC_ExecuteCommands(SC_STATE *sc,char **cmds,char *uout,int chain);

/* convert from SW1SW2 into the card independant rsp codes */
int SC_SW1SW2_RSPCode(SC_STATE *sc,unsigned char SW1,unsigned char SW2);

/* and a table of SW1SW2 independant codes */
char *SC_RSPCode2String(int code);

/* hook into the output parsing logic */
int SC_DoParse(SC_STATE *sc,char *cmd);

/* now for the nice interface into the vars from parsing */
int SC_GetVarCount(SC_STATE *sc);
char *SC_GetVarName(SC_STATE *sc,int idx);
char *SC_GetVarValue(SC_STATE *sc,int idx);
char *SC_GetVar(SC_STATE *sc,char *name);
int SC_SetVar(SC_STATE *sc,char *name,char *val);
int SC_ClearVars(SC_STATE *sc);

/* and handy functions for lazy coders */
int SC_GetIntVar(SC_STATE *sc,char *name);
int SC_SetIntVar(SC_STATE *sc,char *name,int val,int len);

/* simple table of error codes */
#define SC_ERR_NONE                  0
#define SC_ERR_UNSPECIFIED           1
#define SC_ERR_BAD_ATR               2
#define SC_ERR_PROBE_FAILED          3
#define SC_ERR_NO_REPLY              4
#define SC_ERR_NO_INS_ON_REPLY       5
#define SC_ERR_BAD_CMD_FORMAT        6
#define SC_ERR_BAD_CMD_TOO_SHORT     7
#define SC_ERR_BAD_CMD_DATA_MISSING  8
#define SC_ERR_EXE_CMD_NOT_FOUND     9
#define SC_ERR_EXE_NO_CMD_TABLE     10
#define SC_ERR_EXE_MISSING_PARAM    11
#define SC_ERR_MSG_TOO_SHORT        12
#define SC_ERR_WRITE_FAILED         13
#define SC_ERR_READ_FAILED          14
#define SC_ERR_MSG_TRAILER_INVALID  15
#define SC_ERR_MSG_BAD              16
#define SC_ERR_VAR_NO_SPACE         17
#define SC_ERR_VAR_SHORT_DATA       18
#define SC_ERR_PARSE_FAILED         19
#define SC_ERR_BAD_CONVERT_ASCII    20
#define SC_ERR_BAD_CONVERT_VARS     21
#define SC_ERR_MSG_HEADER_INVALID   22
#define SC_ERR_NOT_SUPPORTED        23

#define SC_RSP_OK                    0
#define SC_RSP_UNKNOWN               1
#define SC_RSP_DENIED                2
#define SC_RSP_NOTFOUND              3
#define SC_RSP_BADCLA                4
#define SC_RSP_BADINS                5
#define SC_RSP_BADP1                 6
#define SC_RSP_BADP2                 7
#define SC_RSP_BADP1P2               8
#define SC_RSP_BADLEN                9
#define SC_RSP_BADPASSWORD          10
#define SC_RSP_WRONGTYPE	    11
#define SC_RSP_BLOCKED		    12
#define SC_RSP_NOTSELECTED	    13
#define SC_RSP_BADADDR		    14

#endif /* HEADER_SC_H */

