﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/swf/SWFRequest.h>
#include <aws/swf/SWF_EXPORTS.h>
#include <aws/swf/model/RegistrationStatus.h>

#include <utility>

namespace Aws {
namespace SWF {
namespace Model {

/**
 */
class ListWorkflowTypesRequest : public SWFRequest {
 public:
  AWS_SWF_API ListWorkflowTypesRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "ListWorkflowTypes"; }

  AWS_SWF_API Aws::String SerializePayload() const override;

  AWS_SWF_API Aws::Http::HeaderValueCollection GetRequestSpecificHeaders() const override;

  ///@{
  /**
   * <p>The name of the domain in which the workflow types have been registered.</p>
   */
  inline const Aws::String& GetDomain() const { return m_domain; }
  inline bool DomainHasBeenSet() const { return m_domainHasBeenSet; }
  template <typename DomainT = Aws::String>
  void SetDomain(DomainT&& value) {
    m_domainHasBeenSet = true;
    m_domain = std::forward<DomainT>(value);
  }
  template <typename DomainT = Aws::String>
  ListWorkflowTypesRequest& WithDomain(DomainT&& value) {
    SetDomain(std::forward<DomainT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>If specified, lists the workflow type with this name.</p>
   */
  inline const Aws::String& GetName() const { return m_name; }
  inline bool NameHasBeenSet() const { return m_nameHasBeenSet; }
  template <typename NameT = Aws::String>
  void SetName(NameT&& value) {
    m_nameHasBeenSet = true;
    m_name = std::forward<NameT>(value);
  }
  template <typename NameT = Aws::String>
  ListWorkflowTypesRequest& WithName(NameT&& value) {
    SetName(std::forward<NameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Specifies the registration status of the workflow types to list.</p>
   */
  inline RegistrationStatus GetRegistrationStatus() const { return m_registrationStatus; }
  inline bool RegistrationStatusHasBeenSet() const { return m_registrationStatusHasBeenSet; }
  inline void SetRegistrationStatus(RegistrationStatus value) {
    m_registrationStatusHasBeenSet = true;
    m_registrationStatus = value;
  }
  inline ListWorkflowTypesRequest& WithRegistrationStatus(RegistrationStatus value) {
    SetRegistrationStatus(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>If <code>NextPageToken</code> is returned there are more results available.
   * The value of <code>NextPageToken</code> is a unique pagination token for each
   * page. Make the call again using the returned token to retrieve the next page.
   * Keep all other arguments unchanged. Each pagination token expires after 24
   * hours. Using an expired pagination token will return a <code>400</code> error:
   * "<code>Specified token has exceeded its maximum lifetime</code>". </p> <p>The
   * configured <code>maximumPageSize</code> determines how many results can be
   * returned in a single call. </p>
   */
  inline const Aws::String& GetNextPageToken() const { return m_nextPageToken; }
  inline bool NextPageTokenHasBeenSet() const { return m_nextPageTokenHasBeenSet; }
  template <typename NextPageTokenT = Aws::String>
  void SetNextPageToken(NextPageTokenT&& value) {
    m_nextPageTokenHasBeenSet = true;
    m_nextPageToken = std::forward<NextPageTokenT>(value);
  }
  template <typename NextPageTokenT = Aws::String>
  ListWorkflowTypesRequest& WithNextPageToken(NextPageTokenT&& value) {
    SetNextPageToken(std::forward<NextPageTokenT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The maximum number of results that are returned per call. Use
   * <code>nextPageToken</code> to obtain further pages of results. </p>
   */
  inline int GetMaximumPageSize() const { return m_maximumPageSize; }
  inline bool MaximumPageSizeHasBeenSet() const { return m_maximumPageSizeHasBeenSet; }
  inline void SetMaximumPageSize(int value) {
    m_maximumPageSizeHasBeenSet = true;
    m_maximumPageSize = value;
  }
  inline ListWorkflowTypesRequest& WithMaximumPageSize(int value) {
    SetMaximumPageSize(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>When set to <code>true</code>, returns the results in reverse order. By
   * default the results are returned in ascending alphabetical order of the
   * <code>name</code> of the workflow types.</p>
   */
  inline bool GetReverseOrder() const { return m_reverseOrder; }
  inline bool ReverseOrderHasBeenSet() const { return m_reverseOrderHasBeenSet; }
  inline void SetReverseOrder(bool value) {
    m_reverseOrderHasBeenSet = true;
    m_reverseOrder = value;
  }
  inline ListWorkflowTypesRequest& WithReverseOrder(bool value) {
    SetReverseOrder(value);
    return *this;
  }
  ///@}
 private:
  Aws::String m_domain;

  Aws::String m_name;

  RegistrationStatus m_registrationStatus{RegistrationStatus::NOT_SET};

  Aws::String m_nextPageToken;

  int m_maximumPageSize{0};

  bool m_reverseOrder{false};
  bool m_domainHasBeenSet = false;
  bool m_nameHasBeenSet = false;
  bool m_registrationStatusHasBeenSet = false;
  bool m_nextPageTokenHasBeenSet = false;
  bool m_maximumPageSizeHasBeenSet = false;
  bool m_reverseOrderHasBeenSet = false;
};

}  // namespace Model
}  // namespace SWF
}  // namespace Aws
