/*
 * Decompiled with CFR 0.152.
 */
package org.apache.streampark.console.core.task;

import java.io.File;
import java.io.IOException;
import java.net.URL;
import java.nio.file.CopyOption;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.nio.file.attribute.FileAttribute;
import java.time.Duration;
import java.time.LocalDateTime;
import java.time.format.DateTimeFormatter;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.function.Consumer;
import java.util.regex.Pattern;
import org.apache.commons.lang3.StringUtils;
import org.apache.streampark.common.util.CommandUtils;
import org.apache.streampark.common.util.Utils;
import org.apache.streampark.console.base.util.GitUtils;
import org.apache.streampark.console.core.entity.Project;
import org.apache.streampark.console.core.enums.BuildState;
import org.apache.streampark.console.core.task.AbstractLogFileTask;
import org.eclipse.jgit.api.Git;
import org.eclipse.jgit.lib.StoredConfig;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class ProjectBuildTask
extends AbstractLogFileTask {
    private static final Logger log = LoggerFactory.getLogger(ProjectBuildTask.class);
    private static final Duration CLONE_TIMEOUT = Duration.ofMinutes(10L);
    private static final String BUILD_START_MARKER = "=== BUILD STARTED ===";
    private static final String BUILD_END_MARKER = "=== BUILD COMPLETED ===";
    private static final Pattern SENSITIVE_INFO_PATTERN = Pattern.compile("password|token|key|secret", 2);
    private final Project project;
    private final Consumer<BuildState> stateUpdateConsumer;
    private final Consumer<ch.qos.logback.classic.Logger> notifyReleaseConsumer;
    private final LocalDateTime startTime = LocalDateTime.now();
    private LocalDateTime cloneStartTime;
    private LocalDateTime buildStartTime;
    private LocalDateTime deployStartTime;

    public ProjectBuildTask(String logPath, Project project, Consumer<BuildState> stateUpdateConsumer, Consumer<ch.qos.logback.classic.Logger> notifyReleaseConsumer) {
        super(logPath, true);
        this.project = this.validateProject(project);
        this.stateUpdateConsumer = Objects.requireNonNull(stateUpdateConsumer, "State update consumer cannot be null");
        this.notifyReleaseConsumer = Objects.requireNonNull(notifyReleaseConsumer, "Notify release consumer cannot be null");
        this.validateLogPath(logPath);
        log.info("ProjectBuildTask initialized for project: {} (ID: {})", (Object)project.getName(), (Object)project.getId());
    }

    @Override
    protected void doRun() throws Throwable {
        try {
            this.logBuildStart();
            this.updateBuildState(BuildState.BUILDING);
            if (!this.execute("clone", this::performClone)) {
                this.handleBuildFailure("Failed to clone source code");
                return;
            }
            if (!this.execute("build", this::performBuild)) {
                this.handleBuildFailure("Failed to build project");
                return;
            }
            if (!this.execute("deploy", this::performDeploy)) {
                this.handleBuildFailure("Failed to deploy artifacts");
                return;
            }
            this.handleBuildSuccess();
        }
        catch (Exception e) {
            if (e instanceof InterruptedException) {
                Thread.currentThread().interrupt();
                this.handleBuildInterruption();
            }
            this.handleBuildFailure("Unexpected error during build: " + e.getMessage(), e);
        }
    }

    @Override
    protected void processException(Throwable t) {
        log.error("Build task exception for project: {}", (Object)this.project.getName(), (Object)t);
        this.updateBuildState(BuildState.FAILED);
        if (t instanceof Exception) {
            this.logBuildError("Build failed with exception", (Exception)t);
        } else {
            this.fileLogger.error("Build failed with throwable: {}", (Object)t.getMessage(), (Object)t);
        }
    }

    @Override
    protected void doFinally() {
        try {
            this.cleanupResources();
            this.logBuildSummary();
        }
        catch (Exception e) {
            log.warn("Error during cleanup for project: {}", (Object)this.project.getName(), (Object)e);
        }
    }

    private boolean performClone() throws Exception {
        this.cloneStartTime = LocalDateTime.now();
        try {
            this.validatePreCloneConditions();
            this.prepareCloneDirectory();
            boolean success = this.executeGitClone();
            if (success) {
                this.logCloneSuccess();
            }
            return success;
        }
        catch (Exception e) {
            this.logCloneError("Git clone operation failed", e);
            throw e;
        }
    }

    private void validatePreCloneConditions() throws IllegalStateException {
        if (StringUtils.isBlank((CharSequence)this.project.getUrl())) {
            throw new IllegalStateException("Project URL cannot be blank");
        }
        if (this.project.getAppSource() == null || !this.project.getAppSource().exists()) {
            throw new IllegalStateException("Project source directory does not exist");
        }
        if (!this.project.getUrl().startsWith("http") && !this.project.getUrl().startsWith("git@")) {
            throw new IllegalStateException("Invalid Git URL format: " + this.sanitizeUrl(this.project.getUrl()));
        }
    }

    private void prepareCloneDirectory() throws IOException {
        try {
            this.project.cleanCloned();
            Path sourcePath = this.project.getAppSource().toPath();
            Path parentDir = sourcePath.getParent();
            if (parentDir != null && !Files.exists(parentDir, new LinkOption[0])) {
                Files.createDirectories(parentDir, new FileAttribute[0]);
                this.fileLogger.info("Created parent directory: {}", (Object)parentDir);
            }
        }
        catch (Exception e) {
            throw new IOException("Failed to prepare clone directory: " + e.getMessage(), e);
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private boolean executeGitClone() {
        Git git = null;
        try {
            this.fileLogger.info("Starting Git clone for project: {}", (Object)this.project.getName());
            this.fileLogger.info("Repository URL: {}", (Object)this.sanitizeUrl(this.project.getUrl()));
            this.fileLogger.info("Target directory: {}", (Object)this.project.getAppSource());
            this.fileLogger.info("Branch/Tag: {}", (Object)StringUtils.defaultIfBlank((CharSequence)this.project.getRefs(), (CharSequence)"default"));
            this.fileLogger.info(this.project.getLog4CloneStart());
            GitUtils.GitCloneRequest request = this.buildGitCloneRequest();
            git = GitUtils.clone(request);
            this.configureGitRepository(git);
            this.validateCloneContent(git);
            boolean bl = true;
            return bl;
        }
        catch (Exception e) {
            this.logCloneError("Git clone failed", e);
            boolean bl = false;
            return bl;
        }
        finally {
            if (git != null) {
                try {
                    git.close();
                }
                catch (Exception e) {
                    log.warn("Failed to close Git repository: {}", (Object)e.getMessage());
                }
            }
        }
    }

    private GitUtils.GitCloneRequest buildGitCloneRequest() {
        GitUtils.GitCloneRequest request = new GitUtils.GitCloneRequest();
        request.setUrl(this.project.getUrl());
        request.setRefs(this.project.getRefs());
        request.setStoreDir(this.project.getAppSource());
        request.setUsername(this.project.getUserName());
        request.setPassword(this.project.getPassword());
        request.setPrivateKey(this.project.getPrvkeyPath());
        return request;
    }

    private void configureGitRepository(Git git) throws Exception {
        StoredConfig config = git.getRepository().getConfig();
        String url = this.project.getUrl();
        config.setBoolean("http", url, "sslVerify", false);
        config.setBoolean("https", url, "sslVerify", false);
        config.setInt("http", url, "timeout", (int)CLONE_TIMEOUT.getSeconds());
        config.setInt("https", url, "timeout", (int)CLONE_TIMEOUT.getSeconds());
        config.save();
        this.fileLogger.info("Git repository configuration updated successfully");
    }

    private void validateCloneContent(Git git) {
        File workTree = git.getRepository().getWorkTree();
        if (!workTree.exists() || !workTree.isDirectory()) {
            throw new IllegalStateException("Clone directory does not exist or is not a directory");
        }
        File[] files = workTree.listFiles();
        if (files == null || files.length == 0) {
            throw new IllegalStateException("Cloned repository is empty");
        }
        this.logDirectoryStructure(workTree, "", 0, 3);
        this.fileLogger.info("Clone validation completed. Found {} files/directories", (Object)files.length);
    }

    private boolean performBuild() throws Exception {
        this.buildStartTime = LocalDateTime.now();
        try {
            this.validatePreBuildConditions();
            boolean success = this.executeMavenBuild();
            if (success) {
                Duration buildDuration = Duration.between(this.buildStartTime, LocalDateTime.now());
                this.fileLogger.info("Maven build completed successfully in {}", (Object)this.formatDuration(buildDuration));
            }
            return success;
        }
        catch (Exception e) {
            this.logBuildError("Maven build failed", e);
            throw e;
        }
    }

    private void validatePreBuildConditions() throws IllegalStateException {
        String mavenWorkHome = this.project.getMavenWorkHome();
        if (StringUtils.isBlank((CharSequence)mavenWorkHome)) {
            throw new IllegalStateException("Maven work home cannot be blank");
        }
        File workDir = new File(mavenWorkHome);
        if (!workDir.exists()) {
            throw new IllegalStateException("Maven work directory does not exist: " + mavenWorkHome);
        }
        File pomFile = new File(workDir, "pom.xml");
        File gradleFile = new File(workDir, "build.gradle");
        if (!pomFile.exists() && !gradleFile.exists()) {
            throw new IllegalStateException("No build file (pom.xml or build.gradle) found in: " + mavenWorkHome);
        }
        String mavenArgs = this.project.getMavenBuildArgs();
        if (StringUtils.isBlank((CharSequence)mavenArgs)) {
            throw new IllegalStateException("Maven arguments cannot be blank");
        }
    }

    private boolean executeMavenBuild() {
        try {
            this.fileLogger.info(BUILD_START_MARKER);
            this.fileLogger.info("\ud83d\udd28 Starting Maven build for project: {}", (Object)this.project.getName());
            this.fileLogger.info("   \ud83d\udcc2 Working directory: {}", (Object)this.project.getMavenWorkHome());
            this.fileLogger.info("   \u2699\ufe0f  Build command: {}", (Object)this.sanitizeBuildCommand(this.project.getMavenBuildArgs()));
            int exitCode = CommandUtils.execute((String)this.project.getMavenWorkHome(), Collections.singletonList(this.project.getMavenBuildArgs()), this::logBuildOutput);
            this.fileLogger.info("Maven build completed with exit code: {}", (Object)exitCode);
            this.fileLogger.info(BUILD_END_MARKER);
            return exitCode == 0;
        }
        catch (Exception e) {
            this.fileLogger.error("Maven build execution failed: {}", (Object)e.getMessage());
            return false;
        }
    }

    private void logBuildOutput(String line) {
        if (StringUtils.isNotBlank((CharSequence)line)) {
            String sanitizedLine = this.sanitizeBuildOutput(line);
            this.fileLogger.info(sanitizedLine);
        }
    }

    private boolean performDeploy() throws Exception {
        this.deployStartTime = LocalDateTime.now();
        try {
            this.validatePreDeployConditions();
            this.deployBuildArtifacts();
            this.validateDeploymentResult();
            this.logDeploySuccess();
            return true;
        }
        catch (Exception e) {
            this.logDeployError("Deployment failed", e);
            throw e;
        }
    }

    private void validatePreDeployConditions() throws IllegalStateException {
        File sourceDir = this.project.getAppSource();
        if (sourceDir == null || !sourceDir.exists()) {
            throw new IllegalStateException("Source directory does not exist");
        }
        File distHome = this.project.getDistHome();
        if (distHome == null) {
            throw new IllegalStateException("Distribution home directory is not configured");
        }
    }

    private void deployBuildArtifacts() throws Exception {
        File sourcePath = this.project.getAppSource();
        ArrayList<File> artifacts = new ArrayList<File>();
        this.findBuildArtifacts(artifacts, sourcePath, 0, 5);
        if (artifacts.isEmpty()) {
            throw new RuntimeException("No deployable artifacts (*.tar.gz or *.jar) found in project directory: " + sourcePath.getAbsolutePath());
        }
        this.fileLogger.info("\ud83d\udd0d Found {} deployable artifact(s)", (Object)artifacts.size());
        for (File artifact : artifacts) {
            this.deployArtifact(artifact);
        }
    }

    private void deployArtifact(File artifact) throws Exception {
        String artifactPath = artifact.getAbsolutePath();
        this.fileLogger.info("\ud83d\udce6 Deploying artifact: {}", (Object)artifactPath);
        if (artifactPath.endsWith(".tar.gz")) {
            this.deployTarGzArtifact(artifact);
        } else if (artifactPath.endsWith(".jar")) {
            this.deployJarArtifact(artifact);
        } else {
            throw new IllegalArgumentException("Unsupported artifact type: " + artifactPath);
        }
        this.fileLogger.info("\u2705 Successfully deployed artifact: {}", (Object)artifact.getName());
    }

    private void deployTarGzArtifact(File tarGzFile) throws Exception {
        File deployPath = this.project.getDistHome();
        this.ensureDirectoryExists(deployPath);
        if (!tarGzFile.exists()) {
            throw new IllegalStateException("Tar.gz file does not exist: " + tarGzFile.getAbsolutePath());
        }
        String extractCommand = String.format("tar -xzf %s -C %s", tarGzFile.getAbsolutePath(), deployPath.getAbsolutePath());
        this.fileLogger.info("\ud83d\udce6 Extracting tar.gz: {}", (Object)extractCommand);
        CommandUtils.execute((String)extractCommand);
    }

    private void deployJarArtifact(File jarFile) throws Exception {
        Utils.checkJarFile((URL)jarFile.toURI().toURL());
        String moduleName = jarFile.getName().replace(".jar", "");
        File distHome = this.project.getDistHome();
        File targetDir = new File(distHome, moduleName);
        this.ensureDirectoryExists(targetDir);
        File targetJar = new File(targetDir, jarFile.getName());
        try {
            Files.move(jarFile.toPath(), targetJar.toPath(), new CopyOption[0]);
            this.fileLogger.info("\ud83d\udce6 JAR artifact moved to: {}", (Object)targetJar.getAbsolutePath());
        }
        catch (IOException e) {
            throw new IOException("Failed to move JAR artifact: " + e.getMessage(), e);
        }
    }

    private void validateDeploymentResult() throws Exception {
        File distHome = this.project.getDistHome();
        if (!distHome.exists()) {
            throw new IllegalStateException("Deployment directory was not created");
        }
        File[] deployedFiles = distHome.listFiles();
        if (deployedFiles == null || deployedFiles.length == 0) {
            throw new IllegalStateException("No files were deployed");
        }
        this.fileLogger.info("\u2705 Deployment validation completed. {} items deployed", (Object)deployedFiles.length);
    }

    private boolean execute(String operationName, ExecuteOperation operation) {
        try {
            boolean success = operation.execute();
            if (success) {
                this.fileLogger.info("{} operation succeeded", (Object)operationName);
                return true;
            }
            this.fileLogger.warn("{} operation failed", (Object)operationName);
        }
        catch (InterruptedException e) {
            Thread.currentThread().interrupt();
            this.fileLogger.error("{} operation interrupted", (Object)operationName);
            return false;
        }
        catch (Exception e) {
            this.fileLogger.error("{} operation failed: {}", (Object)operationName, (Object)e.getMessage());
        }
        return false;
    }

    private void findBuildArtifacts(List<File> artifacts, File directory, int currentDepth, int maxDepth) {
        if (currentDepth >= maxDepth || !directory.isDirectory()) {
            return;
        }
        File[] files = directory.listFiles();
        if (files == null) {
            return;
        }
        for (File file : files) {
            if (!file.isDirectory()) continue;
            if ("target".equals(file.getName())) {
                this.findArtifactsInTargetDirectory(artifacts, file);
                continue;
            }
            if (file.getName().startsWith(".")) continue;
            this.findBuildArtifacts(artifacts, file, currentDepth + 1, maxDepth);
        }
    }

    private void findArtifactsInTargetDirectory(List<File> artifacts, File targetDir) {
        File selectedArtifact;
        File[] files = targetDir.listFiles();
        if (files == null) {
            return;
        }
        File tarGzFile = null;
        File jarFile = null;
        for (File file : files) {
            String fileName = file.getName();
            if (fileName.endsWith(".tar.gz")) {
                tarGzFile = file;
                break;
            }
            if (!fileName.endsWith(".jar") || fileName.startsWith("original-") || fileName.endsWith("-sources.jar") || fileName.endsWith("-javadoc.jar") || jarFile != null && file.length() <= jarFile.length()) continue;
            jarFile = file;
        }
        File file = selectedArtifact = tarGzFile != null ? tarGzFile : jarFile;
        if (selectedArtifact != null) {
            artifacts.add(selectedArtifact);
            this.fileLogger.info("\ud83d\udcc4 Found build artifact: {} ({})", (Object)selectedArtifact.getName(), (Object)this.formatFileSize(selectedArtifact.length()));
        }
    }

    private void logDirectoryStructure(File directory, String indent, int currentDepth, int maxDepth) {
        if (currentDepth >= maxDepth || !directory.isDirectory()) {
            return;
        }
        File[] files = directory.listFiles();
        if (files == null) {
            return;
        }
        for (File file : files) {
            if (!file.getName().startsWith(".git") && !file.getName().equals("pom.xml") && !file.getName().equals("build.gradle") && !file.getName().equals("target") && !file.getName().equals("src")) continue;
            String type = file.isDirectory() ? "/" : "";
            this.fileLogger.info("{}\u251c\u2500\u2500 {}{}", new Object[]{indent, file.getName(), type});
            if (!file.isDirectory() || currentDepth >= maxDepth - 1) continue;
            this.logDirectoryStructure(file, indent + "\u2502   ", currentDepth + 1, maxDepth);
        }
    }

    private void updateBuildState(BuildState state) {
        try {
            this.stateUpdateConsumer.accept(state);
        }
        catch (Exception e) {
            log.warn("Failed to update build state to {}: {}", (Object)state, (Object)e.getMessage());
        }
    }

    private void handleBuildSuccess() {
        this.updateBuildState(BuildState.SUCCESSFUL);
        this.fileLogger.info("=== BUILD SUCCESSFUL ===");
        this.fileLogger.info("Project {} built successfully in {}", (Object)this.project.getName(), (Object)this.formatDuration(Duration.between(this.startTime, LocalDateTime.now())));
        try {
            this.notifyReleaseConsumer.accept(this.fileLogger);
        }
        catch (Exception e) {
            log.warn("Failed to notify release consumer: {}", (Object)e.getMessage());
        }
    }

    private void handleBuildFailure(String message) {
        this.handleBuildFailure(message, null);
    }

    private void handleBuildFailure(String message, Throwable cause) {
        this.updateBuildState(BuildState.FAILED);
        this.fileLogger.error("=== BUILD FAILED ===");
        this.fileLogger.error("Project {} build failed: {}", (Object)this.project.getName(), (Object)message);
        if (cause != null) {
            this.fileLogger.error("Cause: {}", (Object)cause.getMessage());
        }
    }

    private void handleBuildInterruption() {
        this.updateBuildState(BuildState.FAILED);
        this.fileLogger.warn("=== BUILD INTERRUPTED ===");
        this.fileLogger.warn("Project {} build was interrupted", (Object)this.project.getName());
    }

    private void logBuildStart() {
        this.fileLogger.info("===============================================");
        this.fileLogger.info("StreamPark Project Build Started");
        this.fileLogger.info("===============================================");
        this.fileLogger.info("Project: {}", (Object)this.project.getName());
        this.fileLogger.info("ID: {}", (Object)this.project.getId());
        this.fileLogger.info("Start Time: {}", (Object)this.startTime.format(DateTimeFormatter.ISO_LOCAL_DATE_TIME));
        this.fileLogger.info("Repository: {}", (Object)this.sanitizeUrl(this.project.getUrl()));
        this.fileLogger.info("Branch/Tag: {}", (Object)StringUtils.defaultIfBlank((CharSequence)this.project.getRefs(), (CharSequence)"default"));
        this.fileLogger.info("Build Args: {}", (Object)this.sanitizeBuildCommand(this.project.getMavenBuildArgs()));
        this.fileLogger.info("===============================================");
        this.fileLogger.info(this.project.getLog4BuildStart());
    }

    private void logBuildSummary() {
        LocalDateTime endTime = LocalDateTime.now();
        Duration totalDuration = Duration.between(this.startTime, endTime);
        this.fileLogger.info("===============================================");
        this.fileLogger.info("Build Summary for Project: {}", (Object)this.project.getName());
        this.fileLogger.info("===============================================");
        this.fileLogger.info("Total Duration: {}", (Object)this.formatDuration(totalDuration));
        this.fileLogger.info("End Time: {}", (Object)endTime.format(DateTimeFormatter.ISO_LOCAL_DATE_TIME));
        this.fileLogger.info("===============================================");
    }

    private void logCloneSuccess() {
        if (this.cloneStartTime != null) {
            Duration cloneDuration = Duration.between(this.cloneStartTime, LocalDateTime.now());
            this.fileLogger.info("Git clone completed successfully in {}", (Object)this.formatDuration(cloneDuration));
            this.fileLogger.info(String.format("[StreamPark] project [%s] git clone successful!", this.project.getName()));
        }
    }

    private void logCloneError(String message, Exception e) {
        this.fileLogger.error("[StreamPark] {}: {}", (Object)message, (Object)e.getMessage());
        this.fileLogger.error("Project: {}, Refs: {}, URL: {}", new Object[]{this.project.getName(), this.project.getRefs(), this.sanitizeUrl(this.project.getUrl())});
    }

    private void logBuildSuccess() {
        if (this.buildStartTime != null) {
            // empty if block
        }
    }

    private void logBuildError(String message, Exception e) {
        this.fileLogger.error("[StreamPark] {}: {}", (Object)message, (Object)e.getMessage());
        this.fileLogger.error("Project: {}, Working Directory: {}", (Object)this.project.getName(), (Object)this.project.getMavenWorkHome());
    }

    private void logDeploySuccess() {
        if (this.deployStartTime != null) {
            Duration deployDuration = Duration.between(this.deployStartTime, LocalDateTime.now());
            this.fileLogger.info("Deployment completed successfully in {}", (Object)this.formatDuration(deployDuration));
        }
    }

    private void logDeployError(String message, Exception e) {
        this.fileLogger.error("[StreamPark] {}: {}", (Object)message, (Object)e.getMessage());
        this.fileLogger.error("Project: {}, Dist Home: {}", (Object)this.project.getName(), (Object)this.project.getDistHome().getAbsolutePath());
    }

    private Project validateProject(Project project) {
        Objects.requireNonNull(project, "Project cannot be null");
        if (project.getId() == null) {
            throw new IllegalArgumentException("Project ID cannot be null");
        }
        if (StringUtils.isBlank((CharSequence)project.getName())) {
            throw new IllegalArgumentException("Project name cannot be blank");
        }
        return project;
    }

    private void validateLogPath(String logPath) {
        if (StringUtils.isBlank((CharSequence)logPath)) {
            throw new IllegalArgumentException("Log path cannot be blank");
        }
        try {
            Path path = Paths.get(logPath, new String[0]);
            Path parent = path.getParent();
            if (parent != null && !Files.exists(parent, new LinkOption[0])) {
                Files.createDirectories(parent, new FileAttribute[0]);
            }
        }
        catch (Exception e) {
            throw new IllegalArgumentException("Invalid log path: " + logPath, e);
        }
    }

    private void ensureDirectoryExists(File directory) throws IOException {
        if (!directory.exists() && !directory.mkdirs()) {
            throw new IOException("Failed to create directory: " + directory.getAbsolutePath());
        }
    }

    private void cleanupResources() {
        try {
            System.gc();
        }
        catch (Exception e) {
            log.debug("Error during resource cleanup: {}", (Object)e.getMessage());
        }
    }

    private String sanitizeUrl(String url) {
        if (StringUtils.isBlank((CharSequence)url)) {
            return "[BLANK_URL]";
        }
        return url.replaceAll("://[^@/]+@", "://***@");
    }

    private String sanitizeBuildCommand(String command) {
        if (StringUtils.isBlank((CharSequence)command)) {
            return "[NO_COMMAND]";
        }
        return SENSITIVE_INFO_PATTERN.matcher(command).replaceAll("***");
    }

    private String sanitizeBuildOutput(String output) {
        if (StringUtils.isBlank((CharSequence)output)) {
            return "";
        }
        return SENSITIVE_INFO_PATTERN.matcher(output).replaceAll("***");
    }

    private String formatDuration(Duration duration) {
        long seconds = duration.getSeconds();
        long minutes = seconds / 60L;
        seconds %= 60L;
        if (minutes > 0L) {
            return String.format("%dm %ds", minutes, seconds);
        }
        return String.format("%ds", seconds);
    }

    private String formatFileSize(long bytes) {
        if (bytes < 1024L) {
            return bytes + " B";
        }
        if (bytes < 0x100000L) {
            return String.format("%.1f KB", (double)bytes / 1024.0);
        }
        return String.format("%.1f MB", (double)bytes / 1048576.0);
    }

    private String truncateString(String str, int maxLength) {
        if (str == null) {
            return "";
        }
        if (str.length() <= maxLength) {
            return str;
        }
        return str.substring(0, Math.max(0, maxLength - 3)) + "...";
    }

    @FunctionalInterface
    private static interface ExecuteOperation {
        public boolean execute() throws Exception;
    }
}

