/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.yarn.server.nodemanager.containermanager.linux.resources.numa;

import java.io.IOException;
import java.io.Serializable;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.thirdparty.com.google.common.annotations.VisibleForTesting;
import org.apache.hadoop.thirdparty.com.google.common.collect.Maps;
import org.apache.hadoop.util.Shell;
import org.apache.hadoop.util.StringUtils;
import org.apache.hadoop.yarn.api.records.ContainerId;
import org.apache.hadoop.yarn.api.records.Resource;
import org.apache.hadoop.yarn.exceptions.YarnException;
import org.apache.hadoop.yarn.server.nodemanager.Context;
import org.apache.hadoop.yarn.server.nodemanager.containermanager.container.Container;
import org.apache.hadoop.yarn.server.nodemanager.containermanager.container.ResourceMappings;
import org.apache.hadoop.yarn.server.nodemanager.containermanager.linux.resources.ResourceHandlerException;
import org.apache.hadoop.yarn.server.nodemanager.containermanager.linux.resources.numa.NumaNodeResource;
import org.apache.hadoop.yarn.server.nodemanager.containermanager.linux.resources.numa.NumaResourceAllocation;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class NumaResourceAllocator {
    private static final Logger LOG = LoggerFactory.getLogger(NumaResourceAllocator.class);
    private static final String NUMA_NODEIDS_REGEX = "available:\\s*[0-9]+\\s*nodes\\s*\\(([0-9\\-,]*)\\)";
    private static final String NUMA_NODE_MEMORY_REGEX = "node\\s*<NUMA-NODE>\\s*size:\\s*([0-9]+)\\s*([KMG]B)";
    private static final String NUMA_NODE_CPUS_REGEX = "node\\s*<NUMA-NODE>\\s*cpus:\\s*([0-9\\s]+)";
    private static final String GB = "GB";
    private static final String KB = "KB";
    private static final String NUMA_NODE = "<NUMA-NODE>";
    private static final String SPACE = "\\s";
    private static final long DEFAULT_NUMA_NODE_MEMORY = 1024L;
    private static final int DEFAULT_NUMA_NODE_CPUS = 1;
    private static final String NUMA_RESOURCE_TYPE = "numa";
    private List<NumaNodeResource> numaNodesList = new ArrayList<NumaNodeResource>();
    private Map<String, NumaNodeResource> numaNodeIdVsResource = new HashMap<String, NumaNodeResource>();
    private int currentAssignNode;
    private Context context;

    public NumaResourceAllocator(Context context) {
        this.context = context;
    }

    public void init(Configuration conf) throws YarnException {
        if (conf.getBoolean("yarn.nodemanager.numa-awareness.read-topology", false)) {
            String[] nodeIdCommaSplits;
            LOG.info("Reading NUMA topology using 'numactl --hardware' command.");
            String cmdOutput = this.executeNGetCmdOutput(conf);
            String[] outputLines = cmdOutput.split("\\n");
            Pattern pattern = Pattern.compile(NUMA_NODEIDS_REGEX);
            String nodeIdsStr = null;
            for (String line : outputLines) {
                Matcher matcher = pattern.matcher(line);
                if (!matcher.find()) continue;
                nodeIdsStr = matcher.group(1);
                break;
            }
            if (nodeIdsStr == null) {
                throw new YarnException("Failed to get numa nodes from 'numactl --hardware' output and output is:\n" + cmdOutput);
            }
            for (String nodeIdOrRange : nodeIdCommaSplits = nodeIdsStr.split("[,\\s]")) {
                if (nodeIdOrRange.contains("-")) {
                    String[] beginNEnd = nodeIdOrRange.split("-");
                    int endNode = Integer.parseInt(beginNEnd[1]);
                    for (int nodeId = Integer.parseInt(beginNEnd[0]); nodeId <= endNode; ++nodeId) {
                        long memory = this.parseMemory(outputLines, String.valueOf(nodeId));
                        int cpus = this.parseCpus(outputLines, String.valueOf(nodeId));
                        this.addToCollection(String.valueOf(nodeId), memory, cpus);
                    }
                    continue;
                }
                long memory = this.parseMemory(outputLines, nodeIdOrRange);
                int cpus = this.parseCpus(outputLines, nodeIdOrRange);
                this.addToCollection(nodeIdOrRange, memory, cpus);
            }
        } else {
            LOG.info("Reading NUMA topology using configurations.");
            Collection nodeIds = conf.getStringCollection("yarn.nodemanager.numa-awareness.node-ids");
            for (String nodeId : nodeIds) {
                long mem = conf.getLong("yarn.nodemanager.numa-awareness." + nodeId + ".memory", 1024L);
                int cpus = conf.getInt("yarn.nodemanager.numa-awareness." + nodeId + ".cpus", 1);
                this.addToCollection(nodeId, mem, cpus);
            }
        }
        if (this.numaNodesList.isEmpty()) {
            throw new YarnException("There are no available NUMA nodes for making containers NUMA aware.");
        }
        LOG.info("Available numa nodes with capacities : " + this.numaNodesList.size());
    }

    @VisibleForTesting
    String executeNGetCmdOutput(Configuration conf) throws YarnException {
        String numaCtlCmd = conf.get("yarn.nodemanager.numa-awareness.numactl.cmd", "/usr/bin/numactl");
        String[] args = new String[]{numaCtlCmd, "--hardware"};
        Shell.ShellCommandExecutor shExec = new Shell.ShellCommandExecutor(args);
        try {
            shExec.execute();
        }
        catch (IOException e) {
            throw new YarnException("Failed to read the numa configurations.", (Throwable)e);
        }
        return shExec.getOutput();
    }

    private int parseCpus(String[] outputLines, String nodeId) {
        int cpus = 0;
        Pattern patternNodeCPUs = Pattern.compile(NUMA_NODE_CPUS_REGEX.replace(NUMA_NODE, nodeId));
        for (String line : outputLines) {
            Matcher matcherNodeCPUs = patternNodeCPUs.matcher(line);
            if (!matcherNodeCPUs.find()) continue;
            String cpusStr = matcherNodeCPUs.group(1);
            cpus = cpusStr.split(SPACE).length;
            break;
        }
        return cpus;
    }

    private long parseMemory(String[] outputLines, String nodeId) throws YarnException {
        long memory = 0L;
        Pattern patternNodeMem = Pattern.compile(NUMA_NODE_MEMORY_REGEX.replace(NUMA_NODE, nodeId));
        for (String line : outputLines) {
            Matcher matcherNodeMem = patternNodeMem.matcher(line);
            if (!matcherNodeMem.find()) continue;
            try {
                memory = Long.parseLong(matcherNodeMem.group(1));
                String units = matcherNodeMem.group(2);
                if (GB.equals(units)) {
                    memory *= 1024L;
                    break;
                }
                if (!KB.equals(units)) break;
                memory /= 1024L;
                break;
            }
            catch (Exception ex) {
                throw new YarnException("Failed to get memory for node:" + nodeId, (Throwable)ex);
            }
        }
        return memory;
    }

    private void addToCollection(String nodeId, long memory, int cpus) {
        NumaNodeResource numaNode = new NumaNodeResource(nodeId, memory, cpus);
        this.numaNodesList.add(numaNode);
        this.numaNodeIdVsResource.put(nodeId, numaNode);
    }

    public synchronized NumaResourceAllocation allocateNumaNodes(Container container) throws ResourceHandlerException {
        NumaResourceAllocation allocation = this.allocate(container.getContainerId(), container.getResource());
        if (allocation != null) {
            try {
                this.context.getNMStateStore().storeAssignedResources(container, NUMA_RESOURCE_TYPE, Arrays.asList(allocation));
            }
            catch (IOException e) {
                this.releaseNumaResource(container.getContainerId());
                throw new ResourceHandlerException(e);
            }
        }
        return allocation;
    }

    private NumaResourceAllocation allocate(ContainerId containerId, Resource resource) {
        for (int index = 0; index < this.numaNodesList.size(); ++index) {
            NumaNodeResource numaNode = this.numaNodesList.get((this.currentAssignNode + index) % this.numaNodesList.size());
            if (!numaNode.isResourcesAvailable(resource)) continue;
            numaNode.assignResources(resource, containerId);
            LOG.info("Assigning NUMA node " + numaNode.getNodeId() + " for memory, " + numaNode.getNodeId() + " for cpus for the " + containerId);
            this.currentAssignNode = (this.currentAssignNode + index + 1) % this.numaNodesList.size();
            return new NumaResourceAllocation(numaNode.getNodeId(), resource.getMemorySize(), numaNode.getNodeId(), resource.getVirtualCores());
        }
        long memoryRequirement = resource.getMemorySize();
        HashMap memoryAllocations = Maps.newHashMap();
        for (NumaNodeResource numaNode : this.numaNodesList) {
            long memoryRemaining = numaNode.assignAvailableMemory(memoryRequirement, containerId);
            memoryAllocations.put(numaNode.getNodeId(), memoryRequirement - memoryRemaining);
            memoryRequirement = memoryRemaining;
            if (memoryRequirement != 0L) continue;
            break;
        }
        if (memoryRequirement != 0L) {
            LOG.info("There is no available memory:" + resource.getMemorySize() + " in numa nodes for " + containerId);
            this.releaseNumaResource(containerId);
            return null;
        }
        int cpusRequirement = resource.getVirtualCores();
        HashMap cpuAllocations = Maps.newHashMap();
        for (int index = 0; index < this.numaNodesList.size(); ++index) {
            NumaNodeResource numaNode = this.numaNodesList.get((this.currentAssignNode + index) % this.numaNodesList.size());
            int cpusRemaining = numaNode.assignAvailableCpus(cpusRequirement, containerId);
            cpuAllocations.put(numaNode.getNodeId(), cpusRequirement - cpusRemaining);
            cpusRequirement = cpusRemaining;
            if (cpusRequirement != 0) continue;
            this.currentAssignNode = (this.currentAssignNode + index + 1) % this.numaNodesList.size();
            break;
        }
        if (cpusRequirement != 0) {
            LOG.info("There are no available cpus:" + resource.getVirtualCores() + " in numa nodes for " + containerId);
            this.releaseNumaResource(containerId);
            return null;
        }
        NumaResourceAllocation assignedNumaNodeInfo = new NumaResourceAllocation(memoryAllocations, cpuAllocations);
        LOG.info("Assigning multiple NUMA nodes (" + StringUtils.join((CharSequence)",", assignedNumaNodeInfo.getMemNodes()) + ") for memory, (" + StringUtils.join((CharSequence)",", assignedNumaNodeInfo.getCpuNodes()) + ") for cpus for " + containerId);
        return assignedNumaNodeInfo;
    }

    public synchronized void releaseNumaResource(ContainerId containerId) {
        LOG.info("Releasing the assigned NUMA resources for " + containerId);
        for (NumaNodeResource numaNode : this.numaNodesList) {
            numaNode.releaseResources(containerId);
        }
    }

    public synchronized void recoverNumaResource(ContainerId containerId) {
        Container container = (Container)this.context.getContainers().get(containerId);
        ResourceMappings resourceMappings = container.getResourceMappings();
        List<Serializable> assignedResources = resourceMappings.getAssignedResources(NUMA_RESOURCE_TYPE);
        if (assignedResources.size() == 1) {
            NumaResourceAllocation numaResourceAllocation = (NumaResourceAllocation)assignedResources.get(0);
            for (Map.Entry nodeAndMemory : numaResourceAllocation.getNodeVsMemory().entrySet()) {
                this.numaNodeIdVsResource.get(nodeAndMemory.getKey()).recoverMemory(containerId, (Long)nodeAndMemory.getValue());
            }
            for (Map.Entry nodeAndCpus : numaResourceAllocation.getNodeVsCpus().entrySet()) {
                this.numaNodeIdVsResource.get(nodeAndCpus.getKey()).recoverCpus(containerId, (Integer)nodeAndCpus.getValue());
            }
        } else {
            LOG.error("Unexpected number:" + assignedResources.size() + " of assigned numa resources for " + containerId + " while recovering.");
        }
    }

    @VisibleForTesting
    Collection<NumaNodeResource> getNumaNodesList() {
        return this.numaNodesList;
    }
}

