/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.hive.ql.io.orc;

import java.io.IOException;
import java.nio.ByteBuffer;
import java.nio.charset.CharacterCodingException;
import java.nio.charset.Charset;
import java.nio.charset.CharsetDecoder;
import java.util.ArrayList;
import java.util.Collection;
import java.util.List;
import org.apache.commons.cli.CommandLine;
import org.apache.commons.cli.GnuParser;
import org.apache.commons.cli.HelpFormatter;
import org.apache.commons.cli.OptionBuilder;
import org.apache.commons.cli.Options;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.fs.FSDataInputStream;
import org.apache.hadoop.fs.FileStatus;
import org.apache.hadoop.fs.FileSystem;
import org.apache.hadoop.fs.Path;
import org.apache.hadoop.hive.ql.io.RecordIdentifier;
import org.apache.hadoop.hive.ql.io.orc.CompressionKind;
import org.apache.hadoop.hive.ql.io.orc.OrcFile;
import org.apache.hadoop.hive.ql.io.orc.OrcInputFormat;
import org.apache.hadoop.hive.ql.io.orc.OrcRecordUpdater;
import org.apache.hadoop.hive.ql.io.orc.OrcStruct;
import org.apache.hadoop.hive.ql.io.orc.Reader;
import org.apache.hadoop.hive.ql.io.orc.RecordReader;
import org.apache.hadoop.hive.ql.io.orc.Writer;
import org.apache.hadoop.hive.serde2.objectinspector.StructField;
import org.apache.hadoop.hive.serde2.objectinspector.StructObjectInspector;
import org.apache.hadoop.hive.serde2.objectinspector.primitive.IntObjectInspector;
import org.apache.hadoop.hive.serde2.objectinspector.primitive.LongObjectInspector;
import org.apache.orc.OrcProto;
import org.apache.orc.StripeInformation;
import org.apache.orc.tools.FileDump;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class FixAcidKeyIndex {
    public static final Logger LOG = LoggerFactory.getLogger(FixAcidKeyIndex.class);
    public static final String DEFAULT_BACKUP_PATH = System.getProperty("java.io.tmpdir");
    private static final Charset UTF8 = Charset.forName("UTF-8");
    private static final CharsetDecoder utf8Decoder = UTF8.newDecoder();

    public static void main(String[] args) throws Exception {
        Configuration conf = new Configuration();
        Options opts = FixAcidKeyIndex.createOptions();
        CommandLine cli = new GnuParser().parse(opts, args);
        if (cli.hasOption('h')) {
            HelpFormatter formatter = new HelpFormatter();
            formatter.printHelp("fixacidkeyindex", opts);
            return;
        }
        String backupPath = DEFAULT_BACKUP_PATH;
        if (cli.hasOption("backup-path")) {
            backupPath = cli.getOptionValue("backup-path");
        }
        boolean checkOnly = cli.hasOption("check-only");
        boolean recover = cli.hasOption("recover");
        String[] files = cli.getArgs();
        if (files.length == 0) {
            System.err.println("Error : ORC files are not specified");
            return;
        }
        ArrayList<String> filesInPath = new ArrayList<String>();
        for (String filename : files) {
            Path path = new Path(filename);
            filesInPath.addAll(FixAcidKeyIndex.getAllFilesInPath(path, conf));
        }
        if (checkOnly) {
            FixAcidKeyIndex.checkFiles(conf, filesInPath);
        } else if (recover) {
            FixAcidKeyIndex.recoverFiles(conf, filesInPath, backupPath);
        } else {
            System.err.println("check-only or recover option must be specified");
        }
    }

    static boolean isAcidKeyIndexValid(Reader reader) {
        if (reader.getNumberOfRows() == 0L) {
            return true;
        }
        List stripes = reader.getStripes();
        RecordIdentifier[] keyIndex = OrcRecordUpdater.parseKeyIndex(reader);
        if (keyIndex == null) {
            return false;
        }
        for (int idx = 0; idx < keyIndex.length; ++idx) {
            if (keyIndex[idx] != null) continue;
            LOG.info("*** keyIndex[" + idx + "] is null");
            return false;
        }
        return stripes.size() == keyIndex.length;
    }

    static void recoverFiles(Configuration conf, List<String> fileList, String backup) {
        for (String fileName : fileList) {
            try {
                Path filePath = new Path(fileName);
                FixAcidKeyIndex.recoverFile(conf, filePath, backup);
            }
            catch (Exception err) {
                System.err.println("ERROR recovering " + fileName);
                err.printStackTrace(System.err);
            }
        }
    }

    static void checkFiles(Configuration conf, List<String> fileList) {
        for (String fileName : fileList) {
            try {
                Path filePath = new Path(fileName);
                FixAcidKeyIndex.checkFile(conf, filePath);
            }
            catch (Exception err) {
                System.err.println("ERROR checking " + fileName);
                err.printStackTrace(System.err);
            }
        }
    }

    static void checkFile(Configuration conf, Path inputPath) throws IOException {
        FileSystem fs = inputPath.getFileSystem(conf);
        Reader reader = OrcFile.createReader(fs, inputPath);
        if (OrcInputFormat.isOriginal(reader)) {
            System.out.println(inputPath + " is not an acid file");
            return;
        }
        boolean validIndex = FixAcidKeyIndex.isAcidKeyIndexValid(reader);
        System.out.println("Checking " + inputPath + " - acid key index is " + (validIndex ? "valid" : "invalid"));
    }

    static void recoverFile(Configuration conf, Path inputPath, String backup) throws IOException {
        FileSystem fs = inputPath.getFileSystem(conf);
        Reader reader = OrcFile.createReader(fs, inputPath);
        if (OrcInputFormat.isOriginal(reader)) {
            System.out.println(inputPath + " is not an acid file. No need to recover.");
            return;
        }
        boolean validIndex = FixAcidKeyIndex.isAcidKeyIndexValid(reader);
        if (validIndex) {
            System.out.println(inputPath + " has a valid acid key index. No need to recover.");
            return;
        }
        System.out.println("Recovering " + inputPath);
        Path recoveredPath = FixAcidKeyIndex.getRecoveryFile(inputPath);
        if (fs.exists(recoveredPath)) {
            fs.delete(recoveredPath, false);
        }
        OrcFile.WriterOptions writerOptions = OrcFile.writerOptions(conf).compress(reader.getCompression()).version(reader.getFileVersion()).rowIndexStride(reader.getRowIndexStride()).inspector(reader.getObjectInspector());
        if (reader.getCompression() != CompressionKind.NONE) {
            writerOptions.bufferSize(reader.getCompressionSize()).enforceBufferSize();
        }
        try (Writer writer = OrcFile.createWriter(recoveredPath, writerOptions);){
            Object stripe;
            String keyIndexString = FixAcidKeyIndex.getKeyIndexAsString(reader);
            if (keyIndexString == null || keyIndexString.equals("null")) {
                keyIndexString = "";
            }
            List stripes = reader.getStripes();
            List stripeStats = reader.getOrcProtoStripeStatistics();
            try (FSDataInputStream inputStream = fs.open(inputPath);){
                for (int idx = 0; idx < stripes.size(); ++idx) {
                    stripe = (StripeInformation)stripes.get(idx);
                    int stripeLength = (int)stripe.getLength();
                    byte[] buffer = new byte[stripeLength];
                    inputStream.readFully(stripe.getOffset(), buffer, 0, stripeLength);
                    writer.appendStripe(buffer, 0, buffer.length, (StripeInformation)stripe, (OrcProto.StripeStatistics)stripeStats.get(idx));
                }
            }
            long lastRow = reader.getNumberOfRows() - 1L;
            RecordReader rr = reader.rows();
            stripe = null;
            try {
                rr.seekToRow(lastRow);
                OrcStruct row = (OrcStruct)rr.next(null);
                StructObjectInspector soi = (StructObjectInspector)reader.getObjectInspector();
                List structFields = soi.getAllStructFieldRefs();
                StructField transactionField = (StructField)structFields.get(1);
                StructField bucketField = (StructField)structFields.get(2);
                StructField rowIdField = (StructField)structFields.get(3);
                long lastTransaction = ((LongObjectInspector)transactionField.getFieldObjectInspector()).get(soi.getStructFieldData((Object)row, transactionField));
                int lastBucket = ((IntObjectInspector)bucketField.getFieldObjectInspector()).get(soi.getStructFieldData((Object)row, bucketField));
                long lastRowId = ((LongObjectInspector)rowIdField.getFieldObjectInspector()).get(soi.getStructFieldData((Object)row, rowIdField));
                keyIndexString = keyIndexString + lastTransaction + "," + lastBucket + "," + lastRowId + ";";
            }
            catch (Throwable throwable) {
                stripe = throwable;
                throw throwable;
            }
            finally {
                if (rr != null) {
                    if (stripe != null) {
                        try {
                            rr.close();
                        }
                        catch (Throwable throwable) {
                            ((Throwable)stripe).addSuppressed(throwable);
                        }
                    } else {
                        rr.close();
                    }
                }
            }
            for (String metadataKey : reader.getMetadataKeys()) {
                if (metadataKey.equals("hive.acid.key.index")) continue;
                writer.addUserMetadata(metadataKey, reader.getMetadataValue(metadataKey));
            }
            writer.addUserMetadata("hive.acid.key.index", UTF8.encode(keyIndexString));
        }
        Reader newReader = OrcFile.createReader(fs, recoveredPath);
        boolean fileFixed = FixAcidKeyIndex.isAcidKeyIndexValid(newReader);
        if (fileFixed) {
            String scheme = inputPath.toUri().getScheme();
            String authority = inputPath.toUri().getAuthority();
            String filePath = inputPath.toUri().getPath();
            Path backupDataPath = backup.equals(DEFAULT_BACKUP_PATH) ? new Path(scheme, authority, DEFAULT_BACKUP_PATH + filePath) : Path.mergePaths((Path)new Path(backup), (Path)inputPath);
            FixAcidKeyIndex.moveFiles(fs, inputPath, backupDataPath);
            FixAcidKeyIndex.moveFiles(fs, recoveredPath, inputPath);
            System.out.println("Fixed acid key index for " + inputPath);
        } else {
            System.out.println("Unable to fix acid key index for " + inputPath);
        }
    }

    private static void moveFiles(FileSystem fs, Path src, Path dest) throws IOException {
        try {
            if (!fs.exists(dest.getParent())) {
                fs.mkdirs(dest.getParent());
            }
            fs.delete(dest, false);
            if (!fs.rename(src, dest)) {
                throw new IOException("Unable to move " + src + " to " + dest);
            }
            System.err.println("Moved " + src + " to " + dest);
        }
        catch (Exception e) {
            throw new IOException("Unable to move " + src + " to " + dest, e);
        }
    }

    static String getKeyIndexAsString(Reader reader) {
        try {
            ByteBuffer val = reader.getMetadataValue("hive.acid.key.index").duplicate();
            return utf8Decoder.decode(val).toString();
        }
        catch (CharacterCodingException e) {
            throw new IllegalArgumentException("Bad string encoding for hive.acid.key.index", e);
        }
    }

    static Path getRecoveryFile(Path corruptPath) {
        return new Path(corruptPath.getParent(), corruptPath.getName() + ".fixacidindex");
    }

    static Options createOptions() {
        Options result = new Options();
        OptionBuilder.withLongOpt((String)"check-only");
        OptionBuilder.withDescription((String)"Check acid orc file for valid acid key index and exit without fixing");
        result.addOption(OptionBuilder.create((char)'c'));
        OptionBuilder.withLongOpt((String)"recover");
        OptionBuilder.withDescription((String)"Fix the acid key index for acid orc file if it requires fixing");
        result.addOption(OptionBuilder.create((char)'r'));
        OptionBuilder.withLongOpt((String)"backup-path");
        OptionBuilder.withDescription((String)"specify a backup path to store the corrupted files (default: /tmp)");
        OptionBuilder.hasArg();
        result.addOption(OptionBuilder.create());
        OptionBuilder.withLongOpt((String)"help");
        OptionBuilder.withDescription((String)"print help message");
        result.addOption(OptionBuilder.create((char)'h'));
        return result;
    }

    public static Collection<String> getAllFilesInPath(Path path, Configuration conf) throws IOException {
        ArrayList<String> filesInPath = new ArrayList<String>();
        FileSystem fs = path.getFileSystem(conf);
        FileStatus fileStatus = fs.getFileStatus(path);
        if (fileStatus.isDir()) {
            FileStatus[] fileStatuses;
            for (FileStatus fileInPath : fileStatuses = fs.listStatus(path, FileDump.HIDDEN_AND_SIDE_FILE_FILTER)) {
                if (fileInPath.isDir()) {
                    filesInPath.addAll(FixAcidKeyIndex.getAllFilesInPath(fileInPath.getPath(), conf));
                    continue;
                }
                filesInPath.add(fileInPath.getPath().toString());
            }
        } else {
            filesInPath.add(path.toString());
        }
        return filesInPath;
    }
}

