/*
 * Decompiled with CFR 0.152.
 */
package org.apache.kafka.common.security.authenticator;

import java.io.Closeable;
import java.io.IOException;
import java.net.InetAddress;
import java.nio.ByteBuffer;
import java.security.PrivilegedActionException;
import java.util.ArrayList;
import java.util.Date;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import javax.security.auth.Subject;
import javax.security.sasl.Sasl;
import javax.security.sasl.SaslException;
import javax.security.sasl.SaslServer;
import org.apache.kafka.common.KafkaException;
import org.apache.kafka.common.errors.AuthenticationException;
import org.apache.kafka.common.errors.IllegalSaslStateException;
import org.apache.kafka.common.errors.InvalidRequestException;
import org.apache.kafka.common.errors.SaslAuthenticationException;
import org.apache.kafka.common.errors.UnsupportedSaslMechanismException;
import org.apache.kafka.common.errors.UnsupportedVersionException;
import org.apache.kafka.common.message.SaslAuthenticateResponseData;
import org.apache.kafka.common.message.SaslHandshakeResponseData;
import org.apache.kafka.common.network.Authenticator;
import org.apache.kafka.common.network.ChannelBuilders;
import org.apache.kafka.common.network.ChannelMetadataRegistry;
import org.apache.kafka.common.network.ClientInformation;
import org.apache.kafka.common.network.ListenerName;
import org.apache.kafka.common.network.NetworkReceive;
import org.apache.kafka.common.network.NetworkSend;
import org.apache.kafka.common.network.ReauthenticationContext;
import org.apache.kafka.common.network.Send;
import org.apache.kafka.common.network.TransportLayer;
import org.apache.kafka.common.protocol.ApiKeys;
import org.apache.kafka.common.protocol.Errors;
import org.apache.kafka.common.requests.AbstractResponse;
import org.apache.kafka.common.requests.ApiVersionsRequest;
import org.apache.kafka.common.requests.ApiVersionsResponse;
import org.apache.kafka.common.requests.RequestAndSize;
import org.apache.kafka.common.requests.RequestContext;
import org.apache.kafka.common.requests.RequestHeader;
import org.apache.kafka.common.requests.SaslAuthenticateRequest;
import org.apache.kafka.common.requests.SaslAuthenticateResponse;
import org.apache.kafka.common.requests.SaslHandshakeRequest;
import org.apache.kafka.common.requests.SaslHandshakeResponse;
import org.apache.kafka.common.security.auth.AuthenticateCallbackHandler;
import org.apache.kafka.common.security.auth.KafkaPrincipal;
import org.apache.kafka.common.security.auth.KafkaPrincipalBuilder;
import org.apache.kafka.common.security.auth.SaslAuthenticationContext;
import org.apache.kafka.common.security.auth.SecurityProtocol;
import org.apache.kafka.common.security.authenticator.SaslClientAuthenticator;
import org.apache.kafka.common.security.kerberos.KerberosError;
import org.apache.kafka.common.security.kerberos.KerberosName;
import org.apache.kafka.common.security.kerberos.KerberosShortNamer;
import org.apache.kafka.common.security.scram.internals.ScramMechanism;
import org.apache.kafka.common.utils.Time;
import org.apache.kafka.common.utils.Utils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class SaslServerAuthenticator
implements Authenticator {
    static final int MAX_RECEIVE_SIZE = 524288;
    private static final Logger LOG = LoggerFactory.getLogger(SaslServerAuthenticator.class);
    private final SecurityProtocol securityProtocol;
    private final ListenerName listenerName;
    private final String connectionId;
    private final Map<String, Subject> subjects;
    private final TransportLayer transportLayer;
    private final List<String> enabledMechanisms;
    private final Map<String, ?> configs;
    private final KafkaPrincipalBuilder principalBuilder;
    private final Map<String, AuthenticateCallbackHandler> callbackHandlers;
    private final Map<String, Long> connectionsMaxReauthMsByMechanism;
    private final Time time;
    private final ReauthInfo reauthInfo;
    private final ChannelMetadataRegistry metadataRegistry;
    private SaslState saslState = SaslState.INITIAL_REQUEST;
    private SaslState pendingSaslState = null;
    private AuthenticationException pendingException = null;
    private SaslServer saslServer;
    private String saslMechanism;
    private NetworkReceive netInBuffer;
    private Send netOutBuffer;
    private Send authenticationFailureSend = null;
    private boolean enableKafkaSaslAuthenticateHeaders;

    public SaslServerAuthenticator(Map<String, ?> configs, Map<String, AuthenticateCallbackHandler> callbackHandlers, String connectionId, Map<String, Subject> subjects, KerberosShortNamer kerberosNameParser, ListenerName listenerName, SecurityProtocol securityProtocol, TransportLayer transportLayer, Map<String, Long> connectionsMaxReauthMsByMechanism, ChannelMetadataRegistry metadataRegistry, Time time) {
        this.callbackHandlers = callbackHandlers;
        this.connectionId = connectionId;
        this.subjects = subjects;
        this.listenerName = listenerName;
        this.securityProtocol = securityProtocol;
        this.enableKafkaSaslAuthenticateHeaders = false;
        this.transportLayer = transportLayer;
        this.connectionsMaxReauthMsByMechanism = connectionsMaxReauthMsByMechanism;
        this.time = time;
        this.reauthInfo = new ReauthInfo();
        this.metadataRegistry = metadataRegistry;
        this.configs = configs;
        List enabledMechanisms = (List)this.configs.get("sasl.enabled.mechanisms");
        if (enabledMechanisms == null || enabledMechanisms.isEmpty()) {
            throw new IllegalArgumentException("No SASL mechanisms are enabled");
        }
        this.enabledMechanisms = new ArrayList<String>(new HashSet(enabledMechanisms));
        for (String mechanism : this.enabledMechanisms) {
            if (!callbackHandlers.containsKey(mechanism)) {
                throw new IllegalArgumentException("Callback handler not specified for SASL mechanism " + mechanism);
            }
            if (!subjects.containsKey(mechanism)) {
                throw new IllegalArgumentException("Subject cannot be null for SASL mechanism " + mechanism);
            }
            LOG.debug("{} for mechanism={}: {}", new Object[]{"connections.max.reauth.ms", mechanism, connectionsMaxReauthMsByMechanism.get(mechanism)});
        }
        this.principalBuilder = ChannelBuilders.createPrincipalBuilder(configs, null, null, kerberosNameParser, null);
    }

    private void createSaslServer(String mechanism) throws IOException {
        this.saslMechanism = mechanism;
        Subject subject = this.subjects.get(mechanism);
        AuthenticateCallbackHandler callbackHandler = this.callbackHandlers.get(mechanism);
        if (mechanism.equals("GSSAPI")) {
            this.saslServer = this.createSaslKerberosServer(callbackHandler, this.configs, subject);
        } else {
            try {
                this.saslServer = Subject.doAs(subject, () -> Sasl.createSaslServer(this.saslMechanism, "kafka", this.serverAddress().getHostName(), this.configs, callbackHandler));
                if (this.saslServer == null) {
                    throw new SaslException("Kafka Server failed to create a SaslServer to interact with a client during session authentication with server mechanism " + this.saslMechanism);
                }
            }
            catch (PrivilegedActionException e) {
                throw new SaslException("Kafka Server failed to create a SaslServer to interact with a client during session authentication with server mechanism " + this.saslMechanism, e.getCause());
            }
        }
    }

    private SaslServer createSaslKerberosServer(AuthenticateCallbackHandler saslServerCallbackHandler, Map<String, ?> configs, Subject subject) throws IOException {
        KerberosName kerberosName;
        String servicePrincipal = SaslClientAuthenticator.firstPrincipal(subject);
        try {
            kerberosName = KerberosName.parse(servicePrincipal);
        }
        catch (IllegalArgumentException e) {
            throw new KafkaException("Principal has name with unexpected format " + servicePrincipal);
        }
        String servicePrincipalName = kerberosName.serviceName();
        String serviceHostname = kerberosName.hostName();
        LOG.debug("Creating SaslServer for {} with mechanism {}", (Object)kerberosName, (Object)this.saslMechanism);
        try {
            return Subject.doAs(subject, () -> Sasl.createSaslServer(this.saslMechanism, servicePrincipalName, serviceHostname, configs, saslServerCallbackHandler));
        }
        catch (PrivilegedActionException e) {
            throw new SaslException("Kafka Server failed to create a SaslServer to interact with a client during session authentication", e.getCause());
        }
    }

    @Override
    public void authenticate() throws IOException {
        if (this.saslState != SaslState.REAUTH_PROCESS_HANDSHAKE) {
            if (this.netOutBuffer != null && !this.flushNetOutBufferAndUpdateInterestOps()) {
                return;
            }
            if (this.saslServer != null && this.saslServer.isComplete()) {
                this.setSaslState(SaslState.COMPLETE);
                return;
            }
            if (this.netInBuffer == null) {
                this.netInBuffer = new NetworkReceive(524288, this.connectionId);
            }
            this.netInBuffer.readFrom(this.transportLayer);
            if (!this.netInBuffer.complete()) {
                return;
            }
            this.netInBuffer.payload().rewind();
        }
        byte[] clientToken = new byte[this.netInBuffer.payload().remaining()];
        this.netInBuffer.payload().get(clientToken, 0, clientToken.length);
        this.netInBuffer = null;
        try {
            switch (this.saslState) {
                case REAUTH_PROCESS_HANDSHAKE: 
                case HANDSHAKE_OR_VERSIONS_REQUEST: 
                case HANDSHAKE_REQUEST: {
                    this.handleKafkaRequest(clientToken);
                    break;
                }
                case REAUTH_BAD_MECHANISM: {
                    throw new SaslAuthenticationException(this.reauthInfo.badMechanismErrorMessage);
                }
                case INITIAL_REQUEST: {
                    if (this.handleKafkaRequest(clientToken)) break;
                }
                case AUTHENTICATE: {
                    this.handleSaslToken(clientToken);
                    if (!this.saslServer.isComplete()) break;
                    this.setSaslState(SaslState.COMPLETE);
                    break;
                }
            }
        }
        catch (AuthenticationException e) {
            this.setSaslState(SaslState.FAILED, e);
        }
        catch (Exception e) {
            this.saslState = SaslState.FAILED;
            LOG.debug("Failed during {}: {}", (Object)this.reauthInfo.authenticationOrReauthenticationText(), (Object)e.getMessage());
            throw e;
        }
    }

    @Override
    public KafkaPrincipal principal() {
        SaslAuthenticationContext context = new SaslAuthenticationContext(this.saslServer, this.securityProtocol, this.clientAddress(), this.listenerName.value());
        KafkaPrincipal principal = this.principalBuilder.build(context);
        if (ScramMechanism.isScram(this.saslMechanism) && Boolean.parseBoolean((String)this.saslServer.getNegotiatedProperty("tokenauth"))) {
            principal.tokenAuthenticated(true);
        }
        return principal;
    }

    @Override
    public boolean complete() {
        return this.saslState == SaslState.COMPLETE;
    }

    @Override
    public void handleAuthenticationFailure() throws IOException {
        this.sendAuthenticationFailureResponse();
    }

    @Override
    public void close() throws IOException {
        if (this.principalBuilder instanceof Closeable) {
            Utils.closeQuietly((Closeable)((Object)this.principalBuilder), "principal builder");
        }
        if (this.saslServer != null) {
            this.saslServer.dispose();
        }
    }

    @Override
    public void reauthenticate(ReauthenticationContext reauthenticationContext) throws IOException {
        NetworkReceive saslHandshakeReceive = reauthenticationContext.networkReceive();
        if (saslHandshakeReceive == null) {
            throw new IllegalArgumentException("Invalid saslHandshakeReceive in server-side re-authentication context: null");
        }
        SaslServerAuthenticator previousSaslServerAuthenticator = (SaslServerAuthenticator)reauthenticationContext.previousAuthenticator();
        this.reauthInfo.reauthenticating(previousSaslServerAuthenticator.saslMechanism, previousSaslServerAuthenticator.principal(), reauthenticationContext.reauthenticationBeginNanos());
        previousSaslServerAuthenticator.close();
        this.netInBuffer = saslHandshakeReceive;
        LOG.debug("Beginning re-authentication: {}", (Object)this);
        this.netInBuffer.payload().rewind();
        this.setSaslState(SaslState.REAUTH_PROCESS_HANDSHAKE);
        this.authenticate();
    }

    @Override
    public Long serverSessionExpirationTimeNanos() {
        return this.reauthInfo.sessionExpirationTimeNanos;
    }

    @Override
    public Long reauthenticationLatencyMs() {
        return this.reauthInfo.reauthenticationLatencyMs();
    }

    @Override
    public boolean connectedClientSupportsReauthentication() {
        return this.reauthInfo.connectedClientSupportsReauthentication;
    }

    private void setSaslState(SaslState saslState) {
        this.setSaslState(saslState, null);
    }

    private void setSaslState(SaslState saslState, AuthenticationException exception) {
        if (this.netOutBuffer != null && !this.netOutBuffer.completed()) {
            this.pendingSaslState = saslState;
            this.pendingException = exception;
        } else {
            this.saslState = saslState;
            LOG.debug("Set SASL server state to {} during {}", (Object)saslState, (Object)this.reauthInfo.authenticationOrReauthenticationText());
            this.pendingSaslState = null;
            this.pendingException = null;
            if (exception != null) {
                throw exception;
            }
        }
    }

    private boolean flushNetOutBufferAndUpdateInterestOps() throws IOException {
        boolean flushedCompletely = this.flushNetOutBuffer();
        if (flushedCompletely) {
            this.transportLayer.removeInterestOps(4);
            if (this.pendingSaslState != null) {
                this.setSaslState(this.pendingSaslState, this.pendingException);
            }
        } else {
            this.transportLayer.addInterestOps(4);
        }
        return flushedCompletely;
    }

    private boolean flushNetOutBuffer() throws IOException {
        if (!this.netOutBuffer.completed()) {
            this.netOutBuffer.writeTo(this.transportLayer);
        }
        return this.netOutBuffer.completed();
    }

    private InetAddress serverAddress() {
        return this.transportLayer.socketChannel().socket().getLocalAddress();
    }

    private InetAddress clientAddress() {
        return this.transportLayer.socketChannel().socket().getInetAddress();
    }

    private void handleSaslToken(byte[] clientToken) throws IOException {
        if (!this.enableKafkaSaslAuthenticateHeaders) {
            byte[] response = this.saslServer.evaluateResponse(clientToken);
            if (this.saslServer.isComplete()) {
                this.reauthInfo.calcCompletionTimesAndReturnSessionLifetimeMs();
                if (this.reauthInfo.reauthenticating()) {
                    this.reauthInfo.ensurePrincipalUnchanged(this.principal());
                }
            }
            if (response != null) {
                this.netOutBuffer = new NetworkSend(this.connectionId, ByteBuffer.wrap(response));
                this.flushNetOutBufferAndUpdateInterestOps();
            }
        } else {
            ByteBuffer requestBuffer = ByteBuffer.wrap(clientToken);
            RequestHeader header = RequestHeader.parse(requestBuffer);
            ApiKeys apiKey = header.apiKey();
            short version = header.apiVersion();
            RequestContext requestContext = new RequestContext(header, this.connectionId, this.clientAddress(), KafkaPrincipal.ANONYMOUS, this.listenerName, this.securityProtocol, ClientInformation.EMPTY);
            RequestAndSize requestAndSize = requestContext.parseRequest(requestBuffer);
            if (apiKey != ApiKeys.SASL_AUTHENTICATE) {
                IllegalSaslStateException e = new IllegalSaslStateException("Unexpected Kafka request of type " + (Object)((Object)apiKey) + " during SASL authentication.");
                this.buildResponseOnAuthenticateFailure(requestContext, requestAndSize.request.getErrorResponse(e));
                throw e;
            }
            if (!apiKey.isVersionSupported(version)) {
                throw new UnsupportedVersionException("Version " + version + " is not supported for apiKey " + (Object)((Object)apiKey));
            }
            if (!this.reauthInfo.connectedClientSupportsReauthentication) {
                this.reauthInfo.connectedClientSupportsReauthentication = version > 0;
            }
            SaslAuthenticateRequest saslAuthenticateRequest = (SaslAuthenticateRequest)requestAndSize.request;
            try {
                byte[] responseToken = this.saslServer.evaluateResponse(Utils.copyArray(saslAuthenticateRequest.data().authBytes()));
                if (this.reauthInfo.reauthenticating() && this.saslServer.isComplete()) {
                    this.reauthInfo.ensurePrincipalUnchanged(this.principal());
                }
                byte[] responseBytes = responseToken == null ? new byte[]{} : responseToken;
                long sessionLifetimeMs = !this.saslServer.isComplete() ? 0L : this.reauthInfo.calcCompletionTimesAndReturnSessionLifetimeMs();
                this.sendKafkaResponse(requestContext, new SaslAuthenticateResponse(new SaslAuthenticateResponseData().setErrorCode(Errors.NONE.code()).setAuthBytes(responseBytes).setSessionLifetimeMs(sessionLifetimeMs)));
            }
            catch (SaslAuthenticationException e) {
                this.buildResponseOnAuthenticateFailure(requestContext, new SaslAuthenticateResponse(new SaslAuthenticateResponseData().setErrorCode(Errors.SASL_AUTHENTICATION_FAILED.code()).setErrorMessage(e.getMessage())));
                throw e;
            }
            catch (SaslException e) {
                KerberosError kerberosError = KerberosError.fromException(e);
                if (kerberosError != null && kerberosError.retriable()) {
                    throw e;
                }
                String errorMessage = "Authentication failed during " + this.reauthInfo.authenticationOrReauthenticationText() + " due to invalid credentials with SASL mechanism " + this.saslMechanism;
                this.buildResponseOnAuthenticateFailure(requestContext, new SaslAuthenticateResponse(new SaslAuthenticateResponseData().setErrorCode(Errors.SASL_AUTHENTICATION_FAILED.code()).setErrorMessage(errorMessage)));
                throw new SaslAuthenticationException(errorMessage, e);
            }
        }
    }

    private boolean handleKafkaRequest(byte[] requestBytes) throws IOException, AuthenticationException {
        boolean isKafkaRequest = false;
        String clientMechanism = null;
        try {
            ByteBuffer requestBuffer = ByteBuffer.wrap(requestBytes);
            RequestHeader header = RequestHeader.parse(requestBuffer);
            ApiKeys apiKey = header.apiKey();
            if (this.saslState == SaslState.INITIAL_REQUEST) {
                this.setSaslState(SaslState.HANDSHAKE_OR_VERSIONS_REQUEST);
            }
            isKafkaRequest = true;
            if (apiKey != ApiKeys.API_VERSIONS && apiKey != ApiKeys.SASL_HANDSHAKE) {
                throw new IllegalSaslStateException("Unexpected Kafka request of type " + (Object)((Object)apiKey) + " during SASL handshake.");
            }
            LOG.debug("Handling Kafka request {} during {}", (Object)apiKey, (Object)this.reauthInfo.authenticationOrReauthenticationText());
            RequestContext requestContext = new RequestContext(header, this.connectionId, this.clientAddress(), KafkaPrincipal.ANONYMOUS, this.listenerName, this.securityProtocol, ClientInformation.EMPTY);
            RequestAndSize requestAndSize = requestContext.parseRequest(requestBuffer);
            if (apiKey == ApiKeys.API_VERSIONS) {
                this.handleApiVersionsRequest(requestContext, (ApiVersionsRequest)requestAndSize.request);
            } else {
                clientMechanism = this.handleHandshakeRequest(requestContext, (SaslHandshakeRequest)requestAndSize.request);
            }
        }
        catch (InvalidRequestException e) {
            if (this.saslState == SaslState.INITIAL_REQUEST) {
                if (LOG.isDebugEnabled()) {
                    StringBuilder tokenBuilder = new StringBuilder();
                    for (byte b : requestBytes) {
                        tokenBuilder.append(String.format("%02x", b));
                        if (tokenBuilder.length() >= 20) break;
                    }
                    LOG.debug("Received client packet of length {} starting with bytes 0x{}, process as GSSAPI packet", (Object)requestBytes.length, (Object)tokenBuilder);
                }
                if (this.enabledMechanisms.contains("GSSAPI")) {
                    LOG.debug("First client packet is not a SASL mechanism request, using default mechanism GSSAPI");
                    clientMechanism = "GSSAPI";
                }
                throw new UnsupportedSaslMechanismException("Exception handling first SASL packet from client, GSSAPI is not supported by server", e);
            }
            throw e;
        }
        if (clientMechanism != null && (!this.reauthInfo.reauthenticating() || this.reauthInfo.saslMechanismUnchanged(clientMechanism))) {
            this.createSaslServer(clientMechanism);
            this.setSaslState(SaslState.AUTHENTICATE);
        }
        return isKafkaRequest;
    }

    private String handleHandshakeRequest(RequestContext context, SaslHandshakeRequest handshakeRequest) throws IOException, UnsupportedSaslMechanismException {
        String clientMechanism = handshakeRequest.data().mechanism();
        short version = context.header.apiVersion();
        if (version >= 1) {
            this.enableKafkaSaslAuthenticateHeaders(true);
        }
        if (this.enabledMechanisms.contains(clientMechanism)) {
            LOG.debug("Using SASL mechanism '{}' provided by client", (Object)clientMechanism);
            this.sendKafkaResponse(context, new SaslHandshakeResponse(new SaslHandshakeResponseData().setErrorCode(Errors.NONE.code()).setMechanisms(this.enabledMechanisms)));
            return clientMechanism;
        }
        LOG.debug("SASL mechanism '{}' requested by client is not supported", (Object)clientMechanism);
        this.buildResponseOnAuthenticateFailure(context, new SaslHandshakeResponse(new SaslHandshakeResponseData().setErrorCode(Errors.UNSUPPORTED_SASL_MECHANISM.code()).setMechanisms(this.enabledMechanisms)));
        throw new UnsupportedSaslMechanismException("Unsupported SASL mechanism " + clientMechanism);
    }

    protected ApiVersionsResponse apiVersionsResponse() {
        return ApiVersionsResponse.DEFAULT_API_VERSIONS_RESPONSE;
    }

    protected void enableKafkaSaslAuthenticateHeaders(boolean flag) {
        this.enableKafkaSaslAuthenticateHeaders = flag;
    }

    private void handleApiVersionsRequest(RequestContext context, ApiVersionsRequest apiVersionsRequest) throws IOException {
        if (this.saslState != SaslState.HANDSHAKE_OR_VERSIONS_REQUEST) {
            throw new IllegalStateException("Unexpected ApiVersions request received during SASL authentication state " + (Object)((Object)this.saslState));
        }
        if (apiVersionsRequest.hasUnsupportedRequestVersion()) {
            this.sendKafkaResponse(context, apiVersionsRequest.getErrorResponse(0, Errors.UNSUPPORTED_VERSION.exception()));
        } else if (!apiVersionsRequest.isValid()) {
            this.sendKafkaResponse(context, apiVersionsRequest.getErrorResponse(0, Errors.INVALID_REQUEST.exception()));
        } else {
            this.metadataRegistry.registerClientInformation(new ClientInformation(apiVersionsRequest.data.clientSoftwareName(), apiVersionsRequest.data.clientSoftwareVersion()));
            this.sendKafkaResponse(context, this.apiVersionsResponse());
            this.setSaslState(SaslState.HANDSHAKE_REQUEST);
        }
    }

    private void buildResponseOnAuthenticateFailure(RequestContext context, AbstractResponse response) {
        this.authenticationFailureSend = context.buildResponse(response);
    }

    private void sendAuthenticationFailureResponse() throws IOException {
        if (this.authenticationFailureSend == null) {
            return;
        }
        this.sendKafkaResponse(this.authenticationFailureSend);
        this.authenticationFailureSend = null;
    }

    private void sendKafkaResponse(RequestContext context, AbstractResponse response) throws IOException {
        this.sendKafkaResponse(context.buildResponse(response));
    }

    private void sendKafkaResponse(Send send) throws IOException {
        this.netOutBuffer = send;
        this.flushNetOutBufferAndUpdateInterestOps();
    }

    private class ReauthInfo {
        public String previousSaslMechanism;
        public KafkaPrincipal previousKafkaPrincipal;
        public long reauthenticationBeginNanos;
        public Long sessionExpirationTimeNanos;
        public boolean connectedClientSupportsReauthentication;
        public long authenticationEndNanos;
        public String badMechanismErrorMessage;

        private ReauthInfo() {
        }

        public void reauthenticating(String previousSaslMechanism, KafkaPrincipal previousKafkaPrincipal, long reauthenticationBeginNanos) {
            this.previousSaslMechanism = Objects.requireNonNull(previousSaslMechanism);
            this.previousKafkaPrincipal = Objects.requireNonNull(previousKafkaPrincipal);
            this.reauthenticationBeginNanos = reauthenticationBeginNanos;
        }

        public boolean reauthenticating() {
            return this.previousSaslMechanism != null;
        }

        public String authenticationOrReauthenticationText() {
            return this.reauthenticating() ? "re-authentication" : "authentication";
        }

        public void ensurePrincipalUnchanged(KafkaPrincipal reauthenticatedKafkaPrincipal) throws SaslAuthenticationException {
            if (!this.previousKafkaPrincipal.equals(reauthenticatedKafkaPrincipal)) {
                throw new SaslAuthenticationException(String.format("Cannot change principals during re-authentication from %s.%s: %s.%s", this.previousKafkaPrincipal.getPrincipalType(), this.previousKafkaPrincipal.getName(), reauthenticatedKafkaPrincipal.getPrincipalType(), reauthenticatedKafkaPrincipal.getName()));
            }
        }

        public boolean saslMechanismUnchanged(String clientMechanism) {
            if (this.previousSaslMechanism.equals(clientMechanism)) {
                return true;
            }
            this.badMechanismErrorMessage = String.format("SASL mechanism '%s' requested by client is not supported for re-authentication of mechanism '%s'", clientMechanism, this.previousSaslMechanism);
            LOG.debug(this.badMechanismErrorMessage);
            SaslServerAuthenticator.this.setSaslState(SaslState.REAUTH_BAD_MECHANISM);
            return false;
        }

        private long calcCompletionTimesAndReturnSessionLifetimeMs() {
            long retvalSessionLifetimeMs = 0L;
            long authenticationEndMs = SaslServerAuthenticator.this.time.milliseconds();
            this.authenticationEndNanos = SaslServerAuthenticator.this.time.nanoseconds();
            Long credentialExpirationMs = (Long)SaslServerAuthenticator.this.saslServer.getNegotiatedProperty("CREDENTIAL.LIFETIME.MS");
            Long connectionsMaxReauthMs = (Long)SaslServerAuthenticator.this.connectionsMaxReauthMsByMechanism.get(SaslServerAuthenticator.this.saslMechanism);
            if ((credentialExpirationMs != null || connectionsMaxReauthMs != null) && (retvalSessionLifetimeMs = credentialExpirationMs == null ? this.zeroIfNegative(connectionsMaxReauthMs) : (connectionsMaxReauthMs == null ? this.zeroIfNegative(credentialExpirationMs - authenticationEndMs) : this.zeroIfNegative(Math.min(credentialExpirationMs - authenticationEndMs, connectionsMaxReauthMs)))) > 0L) {
                this.sessionExpirationTimeNanos = this.authenticationEndNanos + 1000000L * retvalSessionLifetimeMs;
            }
            if (credentialExpirationMs != null) {
                if (this.sessionExpirationTimeNanos != null) {
                    LOG.debug("Authentication complete; session max lifetime from broker config={} ms, credential expiration={} ({} ms); session expiration = {} ({} ms), sending {} ms to client", new Object[]{connectionsMaxReauthMs, new Date(credentialExpirationMs), credentialExpirationMs - authenticationEndMs, new Date(authenticationEndMs + retvalSessionLifetimeMs), retvalSessionLifetimeMs, retvalSessionLifetimeMs});
                } else {
                    LOG.debug("Authentication complete; session max lifetime from broker config={} ms, credential expiration={} ({} ms); no session expiration, sending 0 ms to client", new Object[]{connectionsMaxReauthMs, new Date(credentialExpirationMs), credentialExpirationMs - authenticationEndMs});
                }
            } else if (this.sessionExpirationTimeNanos != null) {
                LOG.debug("Authentication complete; session max lifetime from broker config={} ms, no credential expiration; session expiration = {} ({} ms), sending {} ms to client", new Object[]{connectionsMaxReauthMs, new Date(authenticationEndMs + retvalSessionLifetimeMs), retvalSessionLifetimeMs, retvalSessionLifetimeMs});
            } else {
                LOG.debug("Authentication complete; session max lifetime from broker config={} ms, no credential expiration; no session expiration, sending 0 ms to client", (Object)connectionsMaxReauthMs);
            }
            return retvalSessionLifetimeMs;
        }

        public Long reauthenticationLatencyMs() {
            if (!this.reauthenticating()) {
                return null;
            }
            long latencyNanos = this.authenticationEndNanos - this.reauthenticationBeginNanos;
            return latencyNanos == 0L ? 0L : Math.max(1L, Math.round((double)latencyNanos / 1000.0 / 1000.0));
        }

        private long zeroIfNegative(long value) {
            return Math.max(0L, value);
        }
    }

    private static enum SaslState {
        INITIAL_REQUEST,
        HANDSHAKE_OR_VERSIONS_REQUEST,
        HANDSHAKE_REQUEST,
        AUTHENTICATE,
        COMPLETE,
        FAILED,
        REAUTH_PROCESS_HANDSHAKE,
        REAUTH_BAD_MECHANISM;

    }
}

