/*
 * Decompiled with CFR 0.152.
 */
package org.apache.commons.statistics.descriptive;

import java.math.BigInteger;
import java.util.Objects;
import java.util.Set;
import java.util.function.DoubleConsumer;
import java.util.function.LongConsumer;
import org.apache.commons.statistics.descriptive.FirstMoment;
import org.apache.commons.statistics.descriptive.GeometricMean;
import org.apache.commons.statistics.descriptive.Int128;
import org.apache.commons.statistics.descriptive.Kurtosis;
import org.apache.commons.statistics.descriptive.LongMax;
import org.apache.commons.statistics.descriptive.LongMean;
import org.apache.commons.statistics.descriptive.LongMin;
import org.apache.commons.statistics.descriptive.LongSum;
import org.apache.commons.statistics.descriptive.LongSumOfSquares;
import org.apache.commons.statistics.descriptive.LongVariance;
import org.apache.commons.statistics.descriptive.Product;
import org.apache.commons.statistics.descriptive.RangeFunction;
import org.apache.commons.statistics.descriptive.Skewness;
import org.apache.commons.statistics.descriptive.Statistic;
import org.apache.commons.statistics.descriptive.StatisticResult;
import org.apache.commons.statistics.descriptive.Statistics;
import org.apache.commons.statistics.descriptive.StatisticsConfiguration;
import org.apache.commons.statistics.descriptive.SumOfCubedDeviations;
import org.apache.commons.statistics.descriptive.SumOfFourthDeviations;
import org.apache.commons.statistics.descriptive.SumOfLogs;
import org.apache.commons.statistics.descriptive.UInt192;

public final class LongStatistics
implements LongConsumer {
    private static final String NO_CONFIGURED_STATISTICS = "No configured statistics";
    private static final String UNSUPPORTED_STATISTIC = "Unsupported statistic: ";
    private long count;
    private final LongConsumer consumer;
    private final LongMin min;
    private final LongMax max;
    private final FirstMoment moment;
    private final LongSum sum;
    private final Product product;
    private final LongSumOfSquares sumOfSquares;
    private final SumOfLogs sumOfLogs;
    private StatisticsConfiguration config;

    LongStatistics(long count, LongMin min, LongMax max, FirstMoment moment, LongSum sum, Product product, LongSumOfSquares sumOfSquares, SumOfLogs sumOfLogs, StatisticsConfiguration config) {
        this.count = count;
        this.min = min;
        this.max = max;
        this.moment = moment;
        this.sum = sum;
        this.product = product;
        this.sumOfSquares = sumOfSquares;
        this.sumOfLogs = sumOfLogs;
        this.config = config;
        this.consumer = Statistics.composeLongConsumers(min, max, sum, sumOfSquares, LongStatistics.composeAsLong(moment, product, sumOfLogs));
    }

    private static LongConsumer composeAsLong(DoubleConsumer ... consumers) {
        DoubleConsumer c = Statistics.composeDoubleConsumers(consumers);
        if (c != null) {
            return c::accept;
        }
        return null;
    }

    public static LongStatistics of(Statistic ... statistics) {
        return LongStatistics.builder(statistics).build();
    }

    public static LongStatistics of(Set<Statistic> statistics, long ... values) {
        if (statistics.isEmpty()) {
            throw new IllegalArgumentException(NO_CONFIGURED_STATISTICS);
        }
        Builder b = new Builder();
        statistics.forEach(b::add);
        return b.build(values);
    }

    public static LongStatistics ofRange(Set<Statistic> statistics, long[] values, int from, int to) {
        if (statistics.isEmpty()) {
            throw new IllegalArgumentException(NO_CONFIGURED_STATISTICS);
        }
        Builder b = new Builder();
        statistics.forEach(b::add);
        return b.build(values, from, to);
    }

    public static Builder builder(Statistic ... statistics) {
        if (statistics.length == 0) {
            throw new IllegalArgumentException(NO_CONFIGURED_STATISTICS);
        }
        Builder b = new Builder();
        for (Statistic s : statistics) {
            b.add(s);
        }
        return b;
    }

    @Override
    public void accept(long value) {
        ++this.count;
        this.consumer.accept(value);
    }

    public long getCount() {
        return this.count;
    }

    public boolean isSupported(Statistic statistic) {
        switch (statistic) {
            case GEOMETRIC_MEAN: 
            case SUM_OF_LOGS: {
                return this.sumOfLogs != null;
            }
            case KURTOSIS: {
                return this.moment instanceof SumOfFourthDeviations;
            }
            case MAX: {
                return this.max != null;
            }
            case MIN: {
                return this.min != null;
            }
            case PRODUCT: {
                return this.product != null;
            }
            case SKEWNESS: {
                return this.moment instanceof SumOfCubedDeviations;
            }
            case STANDARD_DEVIATION: 
            case VARIANCE: {
                return this.sum != null && this.sumOfSquares != null;
            }
            case MEAN: 
            case SUM: {
                return this.sum != null;
            }
            case SUM_OF_SQUARES: {
                return this.sumOfSquares != null;
            }
        }
        throw new IllegalArgumentException(UNSUPPORTED_STATISTIC + (Object)((Object)statistic));
    }

    public double getAsDouble(Statistic statistic) {
        return this.getResult(statistic).getAsDouble();
    }

    public long getAsLong(Statistic statistic) {
        return this.getResult(statistic).getAsLong();
    }

    public BigInteger getAsBigInteger(Statistic statistic) {
        return this.getResult(statistic).getAsBigInteger();
    }

    public StatisticResult getResult(Statistic statistic) {
        StatisticResult stat = null;
        switch (statistic) {
            case GEOMETRIC_MEAN: {
                stat = this.getGeometricMean();
                break;
            }
            case KURTOSIS: {
                stat = this.getKurtosis();
                break;
            }
            case MAX: {
                stat = Statistics.getResultAsLongOrNull(this.max);
                break;
            }
            case MEAN: {
                stat = this.getMean();
                break;
            }
            case MIN: {
                stat = Statistics.getResultAsLongOrNull(this.min);
                break;
            }
            case PRODUCT: {
                stat = Statistics.getResultAsDoubleOrNull(this.product);
                break;
            }
            case SKEWNESS: {
                stat = this.getSkewness();
                break;
            }
            case STANDARD_DEVIATION: {
                stat = this.getStandardDeviation();
                break;
            }
            case SUM: {
                stat = Statistics.getResultAsBigIntegerOrNull(this.sum);
                break;
            }
            case SUM_OF_LOGS: {
                stat = Statistics.getResultAsDoubleOrNull(this.sumOfLogs);
                break;
            }
            case SUM_OF_SQUARES: {
                stat = Statistics.getResultAsBigIntegerOrNull(this.sumOfSquares);
                break;
            }
            case VARIANCE: {
                stat = this.getVariance();
            }
        }
        if (stat != null) {
            return stat;
        }
        throw new IllegalArgumentException(UNSUPPORTED_STATISTIC + (Object)((Object)statistic));
    }

    private StatisticResult getGeometricMean() {
        if (this.sumOfLogs != null) {
            return () -> GeometricMean.computeGeometricMean(this.count, this.sumOfLogs);
        }
        return null;
    }

    private StatisticResult getKurtosis() {
        if (this.moment instanceof SumOfFourthDeviations) {
            return new Kurtosis((SumOfFourthDeviations)this.moment).setBiased(this.config.isBiased())::getAsDouble;
        }
        return null;
    }

    private StatisticResult getMean() {
        if (this.sum != null) {
            Int128 s = this.sum.getSum();
            return () -> LongMean.computeMean(s, this.count);
        }
        return null;
    }

    private StatisticResult getSkewness() {
        if (this.moment instanceof SumOfCubedDeviations) {
            return new Skewness((SumOfCubedDeviations)this.moment).setBiased(this.config.isBiased())::getAsDouble;
        }
        return null;
    }

    private StatisticResult getStandardDeviation() {
        return this.getVarianceOrStd(true);
    }

    private StatisticResult getVariance() {
        return this.getVarianceOrStd(false);
    }

    private StatisticResult getVarianceOrStd(boolean std) {
        if (this.sum != null && this.sumOfSquares != null) {
            Int128 s = this.sum.getSum();
            UInt192 ss = this.sumOfSquares.getSumOfSquares();
            boolean biased = this.config.isBiased();
            return () -> LongVariance.computeVarianceOrStd(ss, s, this.count, biased, std);
        }
        return null;
    }

    public LongStatistics combine(LongStatistics other) {
        Statistics.checkCombineCompatible(this.min, other.min);
        Statistics.checkCombineCompatible(this.max, other.max);
        Statistics.checkCombineCompatible(this.sum, other.sum);
        Statistics.checkCombineCompatible(this.product, other.product);
        Statistics.checkCombineCompatible(this.sumOfSquares, other.sumOfSquares);
        Statistics.checkCombineCompatible(this.sumOfLogs, other.sumOfLogs);
        Statistics.checkCombineAssignable(this.moment, other.moment);
        this.count += other.count;
        Statistics.combine(this.min, other.min);
        Statistics.combine(this.max, other.max);
        Statistics.combine(this.sum, other.sum);
        Statistics.combine(this.product, other.product);
        Statistics.combine(this.sumOfSquares, other.sumOfSquares);
        Statistics.combine(this.sumOfLogs, other.sumOfLogs);
        Statistics.combineMoment(this.moment, other.moment);
        return this;
    }

    public LongStatistics setConfiguration(StatisticsConfiguration v) {
        this.config = Objects.requireNonNull(v);
        return this;
    }

    public static final class Builder {
        private static final long[] NO_VALUES = new long[0];
        private RangeFunction<long[], LongMin> min;
        private RangeFunction<long[], LongMax> max;
        private RangeFunction<long[], FirstMoment> moment;
        private RangeFunction<long[], LongSum> sum;
        private RangeFunction<long[], Product> product;
        private RangeFunction<long[], LongSumOfSquares> sumOfSquares;
        private RangeFunction<long[], SumOfLogs> sumOfLogs;
        private int momentOrder;
        private StatisticsConfiguration config = StatisticsConfiguration.withDefaults();

        Builder() {
        }

        Builder add(Statistic statistic) {
            switch (statistic) {
                case GEOMETRIC_MEAN: 
                case SUM_OF_LOGS: {
                    this.sumOfLogs = SumOfLogs::createFromRange;
                    break;
                }
                case KURTOSIS: {
                    this.createMoment(4);
                    break;
                }
                case MAX: {
                    this.max = LongMax::createFromRange;
                    break;
                }
                case MIN: {
                    this.min = LongMin::createFromRange;
                    break;
                }
                case PRODUCT: {
                    this.product = Product::createFromRange;
                    break;
                }
                case SKEWNESS: {
                    this.createMoment(3);
                    break;
                }
                case STANDARD_DEVIATION: 
                case VARIANCE: {
                    this.sum = LongSum::createFromRange;
                    this.sumOfSquares = LongSumOfSquares::createFromRange;
                    break;
                }
                case MEAN: 
                case SUM: {
                    this.sum = LongSum::createFromRange;
                    break;
                }
                case SUM_OF_SQUARES: {
                    this.sumOfSquares = LongSumOfSquares::createFromRange;
                }
            }
            return this;
        }

        private void createMoment(int order) {
            if (order > this.momentOrder) {
                this.momentOrder = order;
                this.moment = order == 4 ? SumOfFourthDeviations::ofRange : SumOfCubedDeviations::ofRange;
            }
        }

        public Builder setConfiguration(StatisticsConfiguration v) {
            this.config = Objects.requireNonNull(v);
            return this;
        }

        public LongStatistics build() {
            return this.create(NO_VALUES, 0, 0);
        }

        public LongStatistics build(long ... values) {
            Objects.requireNonNull(values, "values");
            return this.create(values, 0, values.length);
        }

        public LongStatistics build(long[] values, int from, int to) {
            Statistics.checkFromToIndex(from, to, values.length);
            return this.create(values, from, to);
        }

        private LongStatistics create(long[] values, int from, int to) {
            return new LongStatistics(to - from, Builder.create(this.min, values, from, to), Builder.create(this.max, values, from, to), Builder.create(this.moment, values, from, to), Builder.create(this.sum, values, from, to), Builder.create(this.product, values, from, to), Builder.create(this.sumOfSquares, values, from, to), Builder.create(this.sumOfLogs, values, from, to), this.config);
        }

        private static <S, T> T create(RangeFunction<S, T> constructor, S values, int from, int to) {
            if (constructor != null) {
                return constructor.apply(values, from, to);
            }
            return null;
        }
    }
}

