param(
    [Parameter(Mandatory = $true)]
    [string]$Image,

    [Parameter(Mandatory = $true)]
    [string]$File,

    [string]$OutName
)

Set-StrictMode -Version Latest

# Validate parameters
if (-not $Image) { Write-Error "Image is required" -ErrorAction Stop }
if (-not $File) { Write-Error "File is required" -ErrorAction Stop }
if (-not $OutName) { $OutName = [System.IO.Path]::GetFileNameWithoutExtension($File) }

# Workspace root (host)
$workspace = (Get-Location).ProviderPath

# Try to resolve the file path; handle relative paths sensibly
try {
    $resolved = Resolve-Path -Path $File -ErrorAction Stop
    $fullPath = $resolved.ProviderPath
} catch {
    $fullPath = Join-Path -Path $workspace -ChildPath $File
}

# Compute path relative to workspace when possible
if ($fullPath.StartsWith($workspace, [System.StringComparison]::OrdinalIgnoreCase)) {
    $rel = $fullPath.Substring($workspace.Length).TrimStart('\','/')
} else {
    # If original was relative, use it; otherwise use the full path
    if ($File -match '^[A-Za-z]:\\') { $rel = $fullPath } else { $rel = $File }
}

# Convert backslashes to forward slashes for container
$fileUnix = $rel -replace '\\','/'


# Ensure mount string is safe (quote the workspace path if it contains spaces)
$mount = "$workspace`:/opt/project"

# Build docker args to avoid PowerShell quoting pitfalls
$cmd = "gcc -g '$fileUnix' -o '$OutName'"
$dockerArgs = @('run','--rm','-v',$mount,'-w','/opt/project',$Image,'/bin/sh','-lc',$cmd)
Write-Host "Running: docker $($dockerArgs -join ' ')"

# Sanity check: ensure `docker` is available before attempting to run

# Sanity check: ensure `docker` is available before attempting to run
if (-not (Get-Command docker -ErrorAction SilentlyContinue)) {
    Write-Error "docker executable not found on PATH; aborting." -ErrorAction Stop
}

& docker @dockerArgs
$rc = $LASTEXITCODE
exit $rc
